/**
 * Audit class to perform an async analysis of cookies and external domains
 * 
 * @package GDPR::AUDIT::administrator::components::com:gdpr
 * @subpackage js
 * @author Joomla! Extensions Store
 * @copyright (C) 2019 Joomla! Extensions Store
 * @license GNU/GPLv2 http://www.gnu.org/licenses/gpl-2.0.html
 */
//'use strict';
(function($) {
	var Audit = function() {
		/**
		 * Target audit link, default on the website home page
		 * 
		 * @access private
		 * @var String
		 */
		var targetAuditLink = null;
		
		/**
		 * Register user events for interface controls
		 * 
		 * @access private
		 * @param Boolean initialize
		 * @return Void
		 */
		var addListeners = function(initialize) {
			// Append a dialog for the audit tool
			$('div.audit.icons').on('click.audit', function(jqEvent){
				// Remove any previous instance
				$('#audit_dialog').remove();
				showAuditInterface(gdprBaseAuditHome);
			});
			
			// Bind the start audit button
			$(document).on('click.audit', '#start_analysis', function(jqEvent){
				$('#link_to_analyze').removeClass('error');
				
				// Validate the required field
				targetAuditLink = $('#link_to_analyze').val();
				if(!targetAuditLink) {
					$('#link_to_analyze').addClass('error').focus();
					return false;
				}
				
				// Set the running process interface
				$('span.audit-cogicon', this).addClass('running');
				$('span.audit-labelicon-start', this).text(COM_GDPR_AUDIT_STARTED);
				$('#audit_status').text(COM_GDPR_AUDIT_CURRENT_STATUS_RUNNING).addClass('running');
				
				// Start here the real analysis process
				startAudit(targetAuditLink);
			});
			
			// Bind the start configuration wizard button
			$(document).on('click.audit', '#start_wizard', function(jqEvent){
				// Set the running process interface
				$('span.audit-settingsicon', this).addClass('running');
				$('span.audit-labelicon-wizard', this).text(COM_GDPR_WIZARD_STARTED);
				$('#audit_status').text(COM_GDPR_AUDIT_CURRENT_STATUS_RUNNING).addClass('running');
				
				// Start here the real analysis process
				startAutoConfigurationWizard();
			});
			
			$(document).on('keyup.audit', '#link_to_analyze', function(jqEvent){
				$(this).removeClass('error');
			});
			
			// Closer dialog button
			$(document).on('click.audit', 'label.closedialog', function(jqEvent){
				$(this).parents('#audit_dialog').remove();
			});
			
			// Opener of the analysis link
			$(document).on('click.audit', '#linkopener', function(jqEvent){
				var currentLinkValue = $('#link_to_analyze').val();
				window.open(currentLinkValue, '_blank');
			});
		};
		
		/**
		 * Render the Audit dialog box
		 * 
		 * @access private
		 * @return Void
		 */
		var showAuditInterface = function(linkToAnalyze) {
			var contentsString = '';
			
			// Build modal dialog
			var analysisDialog = '<div id="audit_dialog" class="card card-primary">' +
									'<div class="card-header">' +
								    	'<h3 class="card-title">' + COM_GDPR_AUDIT_DIALOG_TITLE + '</h3>' +
								    	'<label class="closedialog fas fa-times-circle"></label>' +
								    '</div>' +
								    '<div class="card-body card-block">' +
								    	'<label class="badge bg-primary analysis-labels" id="linkopener">' + COM_GDPR_AUDIT_TARGET_LINK + '</label>' +
								    	'<input id="link_to_analyze" class="analysis-input" autocomplete="off" type="text" value="' + linkToAnalyze + '" placeholder="' + COM_GDPR_AUDIT_CHOOSE_TARGET_LINK + '"/>' +
								    	'<button id="start_analysis" class="btn btn-success active">' +
								    		'<span class="audit-cogicon"></span><span class="audit-labelicon-start">' + COM_GDPR_AUDIT_START  + '</span>' +
								    	'</button>' +
								    	'<div class="flexbox-container-analysis">' +
									    	'<div id="cookie_results_container"></div>' +
									    	'<div id="domains_results_container"></div>' +
								    	'</div>' +
								    '</div>' +
								    '<div class="card-footer">' + 
								    	'<span class="badge bg-warning label-with-badge">' + COM_GDPR_AUDIT_FOOTER + '<span id="audit_status" class="badge pull-right">' + COM_GDPR_AUDIT_CURRENT_STATUS_IDLE + '</span></span>' +
								    '</div>' +
								 '</div>';
			// Inject elements into content body
			$('body').append(analysisDialog);
		}

		
		/**
		 * Process the asyncronous audit of the website link that has been specified
		 * It performs an analysis of both cookies (local) and external domains found in the page source code
		 *
		 * @access private
		 * @param String linkToAnalyze
		 * @return Void
		 */
		var startAudit = function(linkToAnalyze) {
			// No ajax request if no valid link and keyword specified
			if(!linkToAnalyze) {
				// Reset the analyzing button
				$('span.audit-cogicon').removeClass('running');
				$('span.audit-labelicon-start').text(COM_GDPR_AUDIT_START);
				$('#audit_status').text(COM_GDPR_AUDIT_CURRENT_STATUS_IDLE).removeClass('running');
				return;
			}
			
			// Clear previous results
			$('label.label-results, ul.analysis-results-cookies, ul.analysis-results-domains').fadeOut(100).promise().done(function(){
				$('#cookie_results_container, #domains_results_container').empty();
			});
			$('#start_wizard').remove();
			
			// Async request
			var auditPromise = $.Deferred(function(defer) {
				$.ajax({
					type : "GET",
					url : linkToAnalyze,
				}).done(function(data, textStatus, jqXHR) {
					// Check response HTTP status code
					defer.resolve(data, jqXHR.status, jqXHR.getAllResponseHeaders());
				}).fail(function(jqXHR, textStatus, errorThrown) {
					// Error found
					defer.resolve(null, jqXHR.status + ' ' + errorThrown);
				});
			}).promise();

			auditPromise.then(function(responseData, status, headers) {
				if(status == 200 && responseData) {
					// All went ok, now analyze and report data
					// STEP 1 - Set the parsed wrapped set
					var responseDataString = responseData.trim();

					/**
					 * STEP 2 Analyze cookies
					 */
					$('#audit_dialog div.card-block #cookie_results_container').append('<label class="badge bg-success analysis-labels label-results">' + COM_GDPR_AUDIT_RESULTS_COOKIE + '</label>');
					var ulCookieResults = $('<ul/>');
					ulCookieResults.addClass('analysis-results-cookies');
					ulCookieResults.append('<li><span>' + COM_GDPR_AUDIT_RESULTS_COOKIE_NAME + '</span><span>' + COM_GDPR_AUDIT_RESULTS_COOKIE_VALUE + '</span></li>');
					
					// STEP 3 - Get the local cookies found
					var addedFoundCookies = false;
					var documentCookies = document.cookie.trim();
					if(documentCookies) {
						var cookies = documentCookies.split("; ");
						// Loop through all cookies
						for (var c = 0; c < cookies.length; c++) {
							var splittedCookie = cookies[c].split("=");
							if(splittedCookie[0] == 'cookieconsent_status') {
								continue;
							}
							ulCookieResults.append(	'<li>' +
															'<span>' +
															'<a class="hasAuditTooltip" title="' + COM_GDPR_AUDIT_SEARCH_COOKIE + '" href="https://www.google.com/search?q=' + encodeURIComponent(splittedCookie[0]) + '" target="_blank">' + splittedCookie[0] + '</a>' +
															'</span>' +
															'<span class="hasCookieValue">' + 
															'<span>' +
															splittedCookie[1] +
															'</span>' +
															'</span>' +
													'</li>');
							addedFoundCookies = true;
						}
					}
					if(!addedFoundCookies) {
						ulCookieResults.append('<li><span>' + COM_GDPR_AUDIT_RESULTS_NOCOOKIES + '</span></li>');
					}
					
					$('#audit_dialog div.card-block #cookie_results_container').append(ulCookieResults);
					
					/**
					 * STEP 4 - Analyze domains/resources
					 */
					$('#audit_dialog div.card-block #domains_results_container').append('<label class="badge bg-success analysis-labels label-results">' + COM_GDPR_AUDIT_RESULTS_DOMAINS + '</label>');
					var ulDomainsResults = $('<ul/>');
					ulDomainsResults.addClass('analysis-results-domains');
					ulDomainsResults.append('<li><span>URL</span></li>');
					
					// Focus keyword regular expression object
					var reObject = new RegExp(/\/\/(www\.)?[-a-zA-Z0-9@:%._\+~#=]{2,256}\.[a-z]{2,6}/, 'gi');
					
					// STEP 5 - Get the external domains/resources of the page
					var addedDomains = [];
					var foundDomains = responseDataString.match(reObject);
					if(foundDomains) {
						// Cycle all found domains
						$.each(foundDomains, function(index, domain) {
							// Exclude domains that are already added and the local domain
							if($.inArray(domain, addedDomains) == -1 && domain.indexOf(gdprBaseDomain) == -1) {
								var purifiedDomain = domain.replace(/^\/\//gi, '');
								ulDomainsResults.append('<li><a class="hasAuditTooltip" title="' + COM_GDPR_AUDIT_SEARCH_DOMAIN + '" href="https://www.google.com/search?q=' + encodeURIComponent(purifiedDomain) + '" target="_blank">' + purifiedDomain + '</a></li>');
								addedDomains[index] = domain;
							}
						});
					}
					
					// Special case for scripts injecting Matomo
					var scriptReObject = new RegExp(/<script[\s\S]*?>[\s\S]*?(matomo.js)[\s\S]*?<\/script>/, 'gi');
					if(responseDataString.match(scriptReObject)) {
						ulDomainsResults.append('<li><a class="hasAuditTooltip" title="' + COM_GDPR_AUDIT_SEARCH_DOMAIN + '" href="https://www.google.com/search?q=' + encodeURIComponent('matomo.js') + '" target="_blank">matomo.js</a></li>');
						addedDomains[addedDomains.length] = 'matomo.js';
					}
					
					if(addedDomains.length == 0) {
						ulDomainsResults.append('<li><span>' + COM_GDPR_AUDIT_RESULTS_NODOMAINS + '</span></li>');
					}
					
					$('#audit_dialog div.card-block #domains_results_container').append(ulDomainsResults);
					
					// Add the button to start the auto configuration wizard
					$('#start_analysis').next('#start_wizard').remove();
					$('#start_analysis').after(	'<button id="start_wizard" class="btn btn-primary active" title="' + COM_GDPR_WIZARD_DESCRIPTION + '">' +
													'<span class="audit-settingsicon"></span><span class="audit-labelicon-wizard">' + COM_GDPR_WIZARD_START  + '</span>' +
												'</button>');
					new bootstrap.Tooltip(document.querySelector('#start_wizard'), {
						trigger:'hover', 
						placement:'top', 
						html : true, 
						container : '#audit_dialog'
					});
				} else {
					// An error in the AJAX request occurred, kindly inform user and return
					$('#cookie_results_container').append('<span class="badge bg-danger"><span class="icon-warning"></span> ' + COM_GDPR_AUDIT_AUDIT_ERROR + status + '</span>');
					return;
				}
			}).always(function(){
				// Reset the analyzing button
				$('span.audit-cogicon').removeClass('running');
				$('span.audit-labelicon-start').text(COM_GDPR_AUDIT_START);
				$('#audit_status').text(COM_GDPR_AUDIT_CURRENT_STATUS_IDLE).removeClass('running');
				
				// Initialize tooltips of the audit results
				[].slice.call(document.querySelectorAll('a.hasAuditTooltip')).map(function (tooltipEl) {
					return new bootstrap.Tooltip(tooltipEl, {
						trigger:'hover', 
						placement:'top', 
						html : true, 
						container : '#audit_dialog'
					});
				});
			});
		};

		/**
		 * The first operation is get informations about published data sources
		 * and start cycle over all the records using promises and recursion
		 * 
		 * @access private
		 * @return Void
		 */
		var startAutoConfigurationWizard = function() {
			// Retrieve cookies found by the audit, put them in the posted object
			var cookiesAudited = [];
			var cookiesAuditedHref = $('ul.analysis-results-cookies a');
			cookiesAuditedHref.each(function(index, cookieName) {
				cookiesAudited[index] = $(cookieName).text();
			});

			// Retrieve domains found by the audit, put them in the posted object
			var domainsAudited = [];
			var domainsAuditedHref = $('ul.analysis-results-domains a');
			domainsAuditedHref.each(function(index, domainName) {
				domainsAudited[index] = $(domainName).text();
			});

			// Clear previous results
			$('label.label-results, ul.analysis-results-cookies, ul.analysis-results-domains').fadeOut(100).promise().done(function(){
				$('#cookie_results_container, #domains_results_container').empty();
			});
			
			// Object to send to server
			var ajaxparams = {
			    idtask : 'autoConfigurationWizard',
			    template : 'json',
			    param : {
			        'cookies' : cookiesAudited,
			        'domains' : domainsAudited
			    }
			};

			// Unique param 'data'
			var uniqueParam = JSON.stringify(ajaxparams);
			// Request JSON2JSON
			var summaryPromise = $.Deferred(function(defer) {
				$.ajax({
					type : "POST",
					url : "../administrator/index.php?option=com_gdpr&task=ajaxserver.display&format=json",
					dataType : 'json',
					context : this,
					data : {
						data : uniqueParam
					}
				}).done(function(data, textStatus, jqXHR) {
					if(data === null) {
						// Error found
						defer.reject(COM_GDPR_WIZARD_NULL_RESPONSEDATA, 'warning');
						return false;
					}
					
					if(!data.result) {
						// Error found
						defer.reject(data.exception_message, data.errorlevel, data.summary, textStatus);
						return false;
					}
					
					// Check response all went well
					if(data.result && data.summary) {
						defer.resolve(data.summary);
					}
				}).fail(function(jqXHR, textStatus, errorThrown) {
					// Error found
					var genericStatus = textStatus[0].toUpperCase() + textStatus.slice(1) + ': ' + jqXHR.status ;
					defer.reject(COM_GDPR_WIZARD_ERROR_HTTP + '-' + genericStatus + '- ' + errorThrown, 'danger');
				});
			}).promise();

			summaryPromise.then(function(responseData) {
				$('#cookie_results_container, #domains_results_container').empty();
				
				/**
				 * Append found matched cookies to the database
				 */
				$('#audit_dialog div.card-body #cookie_results_container').append('<label class="badge bg-success analysis-labels label-results">' + COM_GDPR_WIZARD_RESULTS_COOKIE + '</label>');
				var ulCookieResults = $('<ul/>');
				ulCookieResults.addClass('analysis-results-cookies');
				ulCookieResults.append('<li><span>' + COM_GDPR_WIZARD_COOKIE_RESULTS_NAME + '</span><span>' + COM_GDPR_WIZARD_RESULTS_CATEGORY + '</span></li>');
				
				var addedFoundCookies = false;
				if(responseData.foundCookies) {
					// Loop through all cookies
					for (var c = 0; c < responseData.foundCookies.length; c++) {
						ulCookieResults.append(	'<li>' +
														'<span>' +
														'<a class="hasAuditTooltip" title="' + COM_GDPR_AUDIT_SEARCH_COOKIE + '" href="https://www.google.com/search?q=' + encodeURIComponent(responseData.foundCookies[c][0]) + '" target="_blank">' + responseData.foundCookies[c][0] + '</a>' +
														'</span>' +
														'<span class="hasCookieValue">' + 
														'<span>' +
														responseData.foundCookies[c][1] +
														'</span>' +
														'</span>' +
												'</li>');
						addedFoundCookies = true;
					}
				}
				if(!addedFoundCookies) {
					ulCookieResults.append('<li><span>' + COM_GDPR_WIZARD_RESULTS_NOCOOKIES + '</span></li>');
				}
				
				$('#audit_dialog div.card-body #cookie_results_container').append(ulCookieResults);
				
				/**
				 * Append found matched domains to the database
				 */
				$('#audit_dialog div.card-body #domains_results_container').append('<label class="badge bg-success analysis-labels label-results">' + COM_GDPR_WIZARD_RESULTS_DOMAINS + '</label>');
				var ulDomainsResults = $('<ul/>');
				ulDomainsResults.addClass('analysis-results-domains');
				ulDomainsResults.append('<li><span>' + COM_GDPR_WIZARD_DOMAIN_RESULTS_NAME + '</span><span>' + COM_GDPR_WIZARD_RESULTS_CATEGORY + '</span></li>');
				
				// Focus keyword regular expression object
				var reObject = new RegExp(/\/\/(www\.)?[-a-zA-Z0-9@:%._\+~#=]{2,256}\.[a-z]{2,6}/, 'gi');
				
				// STEP 5 - Get the external domains/resources of the page
				var addedDomains = [];
				if(responseData.foundDomains) {
					// Cycle all found domains
					$.each(responseData.foundDomains, function(index, domain) {
						// Exclude domains that are already added and the local domain
						var purifiedDomain = domain[0].replace(/^\/\//gi, '');
						ulDomainsResults.append('<li>' +
													'<span>' +
													'<a class="hasAuditTooltip" title="' + COM_GDPR_AUDIT_SEARCH_DOMAIN + '" href="https://www.google.com/search?q=' + encodeURIComponent(purifiedDomain) + '" target="_blank">' + purifiedDomain + '</a>' +
													'</span>' +
													'<span>' +
													domain[1] +
													'</span>' +
													'</li>');
						addedDomains[index] = domain;
					});
				}
				if(addedDomains.length == 0) {
					ulDomainsResults.append('<li><span>' + COM_GDPR_WIZARD_RESULTS_NODOMAINS + '</span></li>');
				}
				
				$('#audit_dialog div.card-body #domains_results_container').append(ulDomainsResults);
				
			}, function(errorText, errorLevel, responseData, error) {
				$('#cookie_results_container').append('<span class="badge bg-' + errorLevel + '">' + errorText + '</span>');
			}).always(function(){
				// Reset the analyzing button
				$('span.audit-settingsicon').removeClass('running');
				$('span.audit-labelicon-wizard').text(COM_GDPR_WIZARD_START);
				$('#audit_status').text(COM_GDPR_AUDIT_CURRENT_STATUS_IDLE).removeClass('running');
				
				// Initialize tooltips of the audit results
				[].slice.call(document.querySelectorAll('a.hasAuditTooltip')).map(function (tooltipEl) {
					return new bootstrap.Tooltip(tooltipEl, {
						trigger:'hover', 
						placement:'top', 
						html : true, 
						container : '#audit_dialog'
					});
				});
			})
		};
		
		/**
		 * Function dummy constructor
		 * 
		 * @access private
		 * @param String
		 *            contextSelector
		 * @method <<IIFE>>
		 * @return Void
		 */
		(function __construct() {
			// Add UI events
			addListeners.call(this, true);
		}).call(this);
	}

	// On DOM Ready
	$(function() {
		window.GDPRAudit = new Audit();
	});
})(jQuery);