<?php
namespace JExtstore\Component\Gdpr\Administrator\Model;
/**
 *
 * @package GDPR::CPANEL::administrator::components::com_gdpr
 * @subpackage models
 * @author Joomla! Extensions Store
 * @copyright (C) 2018 - Joomla! Extensions Store
 * @license GNU/GPLv2 http://www.gnu.org/licenses/gpl-2.0.html
 */
defined ( '_JEXEC' ) or die ( 'Restricted access' );
define ( 'SERVER_REMOTE_URI', 'http://storejextensions.org/dmdocuments/updates/' );
define ( 'UPDATES_FORMAT', '.json' );

use Joomla\CMS\MVC\Factory\MVCFactoryInterface;
use Joomla\CMS\Language\Text;
use JExtstore\Component\Gdpr\Administrator\Framework\Http;
use JExtstore\Component\Gdpr\Administrator\Framework\Model as GdprModel;
use JExtstore\Component\Gdpr\Administrator\Framework\Exception as GdprException;

/**
 * Messages model responsibilities contract
 *
 * @package GDPR::MESSAGES::administrator::components::com_gdpr
 * @subpackage models
 * @since 1.6
 */
interface ICPanelModel {
	/**
	 * Get by remote server informations for new updates of this extension
	 *
	 * @access public
	 * @param Http $httpClient        	
	 * @return mixed An object json decoded from server if update information retrieved correctly otherwise false
	 */
	public function getUpdates(Http $httpClient);
	
	/**
	 * Delete from file system all obsolete exchanged files
	 * 
	 * @access public
	 * @return boolean
	 */
	public function purgeFileCache();
}
/**
 * CPanel model concrete implementation
 *
 * @package GDPR::CPANEL::administrator::components::com_gdpr
 * @subpackage models
 * @since 1.6
 */
class CpanelModel extends GdprModel {
	/**
	 * Build list entities query
	 *
	 * @access protected
	 * @return string
	 */
	protected function buildListQuery() {
		$where = null;
		
		// Skip pseudonymised users if any and the inclusion is not enabled
		if(!$this->getComponentParams()->get('consent_registry_include_pseudonymised', 0)) {
			$where = "\n AND u.name != ''";
		}
		
		if(!$this->getComponentParams()->get('consent_registry_include_pseudonymised', 0)) {
			$where .= "\n AND SUBSTRING(u.email, -7) != " .  $this->dbInstance->quote('invalid');
		}
		
		$query = "SELECT u.name, u.username, u.email, registerDate, p.profile_value, n.body" .
				 "\n FROM #__user_profiles AS p" .
				 "\n INNER JOIN #__users as u ON p.user_id = u.id" .
				 "\n LEFT JOIN #__user_notes AS n ON p.user_id = n.user_id AND n.state = 1" .
				 "\n AND n.subject = " . $this->dbInstance->quote(Text::_('COM_GDPR_PRIVACY_ACCEPTED_SUBJECT')) .
				 "\n WHERE p.profile_key = " . $this->dbInstance->quote('gdpr_consent_status') .
				 $where .
				 "\n ORDER BY p.user_id DESC";

		return $query;
	}
	
	/**
	 * Counter result set
	 *
	 * @access protected
	 * @return int
	 */
	protected function buildListQueryNewUsers() {
		$query = "SELECT COUNT(*)" .
				 "\n FROM #__gdpr_logs AS s" .
				 "\n WHERE s.created_user = 1";
	
		return $query;
	}
	
	/**
	 * Counter result set
	 *
	 * @access protected
	 * @return int
	 */
	protected function buildListQueryDeletedUsers() {
		$query = "SELECT COUNT(*)" .
				 "\n FROM #__gdpr_logs AS s" .
				 "\n WHERE s.deleted_user = 1";
	
		return $query;
	}
	
	/**
	 * Counter result set
	 *
	 * @access protected
	 * @return int
	 */
	protected function buildListQueryBreachedUsers() {
		$query = "SELECT COUNT(*)" .
				 "\n FROM #__gdpr_databreach_users AS s" .
				 "\n WHERE s.violated_user = 1";
	
		return $query;
	}
	
	/**
	 * Counter result set
	 *
	 * @access protected
	 * @param int $categoryId
	 * @return int
	 */
	protected function buildListQueryAllUsers() {
		$query = "SELECT COUNT(*)" .
				 "\n FROM #__gdpr_cookie_consent_registry";
		
		return $query;
	}
	
	/**
	 * Counter result set
	 *
	 * @access protected
	 * @param int $categoryId
	 * @return int
	 */
	protected function buildListQueryCookieCategory($categoryId) {
		$query = "SELECT COUNT(*)" .
				 "\n FROM #__gdpr_cookie_consent_registry AS s" .
				 "\n WHERE s.category$categoryId = 1";
		
		return $query;
	}
	
	/**
	 * Counter result set
	 *
	 * @access protected
	 * @param int $categoryId
	 * @return string
	 */
	protected function buildListQueryCookiePercentage() {
		$dateNowFormatted = date('Y-m-d');
		$timeFrom = strtotime ( "-1 month", strtotime ( $dateNowFormatted ) );
		$startFrom = date ( "Y-m-d", $timeFrom );
		$startFromDate = "\n s.consent_date > " . $this->dbInstance->quote($startFrom);
		
		$query = "SELECT COUNT(*)" .
				 "\n FROM #__gdpr_cookie_consent_registry AS s" .
				 "\n WHERE " . $startFromDate .
				 "\n AND s.generic = 1";
		$this->dbInstance->setQuery ( $query );
		$totalAcceptedSessions = $this->dbInstance->loadResult ();
		
		$query = "SELECT COUNT(*)" .
				 "\n FROM #__session AS sess" .
				 "\n WHERE (sess.client_id = 0 OR ISNULL(sess.client_id))" .
				 "\n AND sess.time > " . $timeFrom;
		$this->dbInstance->setQuery ( $query );
		$totalRecordedSessions = $this->dbInstance->loadResult ();
		
		if($totalRecordedSessions < $totalAcceptedSessions) {
			$totalRecordedSessions = $totalAcceptedSessions;
		}
		if($totalRecordedSessions == 0) {
			$totalRecordedSessions = 1;
		}
		$calculatedPercentage = number_format(($totalAcceptedSessions / $totalRecordedSessions) * 100, 1) . '%';
		
		return $calculatedPercentage;
	}
	
	/**
	 * Counter result set
	 *
	 * @access protected
	 * @return int
	 */
	protected function buildListQueryAcceptedAll() {
		$where = array();
		for($i = 1, $k = 4; $i <= $k; $i ++) {
			if ($this->getComponentParams ()->get ( 'cookie_category' . $i . '_enable', 0 )) {
				$where [] = "\n s.category$i = 1";
			}
		}
		
		$query = "SELECT COUNT(*)" .
				 "\n FROM #__gdpr_cookie_consent_registry AS s" .
				 "\n WHERE s.generic = 1";
		if(count($where)) {
			$query .= "\n AND " . implode(' AND ', $where);
		}
		
		return $query;
	}
	
	/**
	 * Main get data method
	 *
	 * @access public
	 * @return array
	 */
	public function getData(): array {
		$calculatedStats = array ();
		// Build queries
		try {
			// All Users
			$query = $this->buildListQueryAllUsers();
			$this->dbInstance->setQuery ( $query );
			$totalSessionsUsers = $this->dbInstance->loadResult ();
			// Avoid division by zero
			if($totalSessionsUsers == 0) {
				$totalSessionsUsers = 1;
			}
			
			// New Users
			$query = $this->buildListQueryNewUsers();
			$this->dbInstance->setQuery ( $query );
			$totalNewUsers = $this->dbInstance->loadResult ();
			
			// Deleted Users
			$query = $this->buildListQueryDeletedUsers();
			$this->dbInstance->setQuery ( $query );
			$totalDeletedUsers = $this->dbInstance->loadResult ();
			
			// New Users
			$query = $this->buildListQueryBreachedUsers();
			$this->dbInstance->setQuery ( $query );
			$totalBreachedUsers = $this->dbInstance->loadResult ();
			
			if($this->getComponentParams()->get('enable_log_cookie_consent', 1)) {
				// Users that have accepted cookie category 1
				if($this->getComponentParams()->get('cookie_category1_enable', 0)) {
					$query = $this->buildListQueryCookieCategory(1);
					$this->dbInstance->setQuery ( $query );
					$totalUsers = $this->dbInstance->loadResult ();
					$calculatedStats ['chart_gdpr_cookies_canvas']['category1'] = number_format(($totalUsers / $totalSessionsUsers) * 100, 1);
				}
				
				// Users that have accepted cookie category 2
				if($this->getComponentParams()->get('cookie_category2_enable', 0)) {
					$query = $this->buildListQueryCookieCategory(2);
					$this->dbInstance->setQuery ( $query );
					$totalUsers = $this->dbInstance->loadResult ();
					$calculatedStats ['chart_gdpr_cookies_canvas']['category2'] = number_format(($totalUsers / $totalSessionsUsers) * 100, 1);
				}
				
				// Users that have accepted cookie category 3
				if($this->getComponentParams()->get('cookie_category3_enable', 0)) {
					$query = $this->buildListQueryCookieCategory(3);
					$this->dbInstance->setQuery ( $query );
					$totalUsers = $this->dbInstance->loadResult ();
					$calculatedStats ['chart_gdpr_cookies_canvas']['category3'] = number_format(($totalUsers / $totalSessionsUsers) * 100, 1);
				}
				
				// Users that have accepted cookie category 4
				if($this->getComponentParams()->get('cookie_category4_enable', 0)) {
					$query = $this->buildListQueryCookieCategory(4);
					$this->dbInstance->setQuery ( $query );
					$totalUsers = $this->dbInstance->loadResult ();
					$calculatedStats ['chart_gdpr_cookies_canvas']['category4'] = number_format(($totalUsers / $totalSessionsUsers) * 100, 1);
				}
				
				// Percentage of sessions with cookies accepted
				$calculatedStats ['off_canvas']['total_cookie_percentage'] = $this->buildListQueryCookiePercentage();
				
				// Percentage of session with accept all cookies/categories
				$query = $this->buildListQueryAcceptedAll();
				$this->dbInstance->setQuery ( $query );
				$totalUsers = $this->dbInstance->loadResult ();
				$calculatedStats ['off_canvas']['all_cookies_categories_accepted'] = number_format(($totalUsers / $totalSessionsUsers) * 100, 1) . '%';
			}
			
			// GPlus registered users
			$calculatedStats ['chart_gdpr_canvas'] ['new'] = $totalNewUsers;
			$calculatedStats ['chart_gdpr_canvas'] ['deleted'] = $totalDeletedUsers;
			$calculatedStats ['chart_gdpr_canvas'] ['breached'] = $totalBreachedUsers;
		} catch ( GdprException $e ) {
			$this->app->enqueueMessage ( $e->getMessage (), $e->getExceptionLevel () );
			$calculatedStats = array ();
		} catch ( \Exception $e ) {
			$gdprException = new GdprException ( $e->getMessage (), 'error' );
			$this->app->enqueueMessage ( $gdprException->getMessage (), $gdprException->getExceptionLevel () );
			$calculatedStats = array ();
		}
		
		return $calculatedStats;
	}
	
	/**
	 * Esplica la funzione di esportazione del registro in formato CSV
	 * @access public
	 * @return Object[]&
	 */
	public function exportRegistry($dataType = 'assoc_array') {
		try {
			// Obtain query string
			$query = $this->buildListQuery();
			$this->dbInstance->setQuery($query);
			if($dataType == 'assoc_array') {
				$resultSet = $this->dbInstance->loadAssocList();
			} else {
				$resultSet = $this->dbInstance->loadObjectList();
			}
		} catch ( GdprException $e ) {
			$this->app->enqueueMessage ( $e->getMessage (), $e->getExceptionLevel () );
			$resultSet = array ();
		} catch ( \Exception $e ) {
			$gdprException = new GdprException ( $e->getMessage (), 'error' );
			$this->app->enqueueMessage ( $gdprException->getMessage (), $gdprException->getExceptionLevel () );
			$resultSet = array ();
		}
	
		if(!is_array($resultSet) || !count($resultSet)) {
			return false;
		}
	
		return $resultSet;
	}
	
	/**
	 * Get by remote server informations for new updates of this extension
	 *
	 * @access public
	 * @param Http $httpClient        	
	 * @return mixed An object json decoded from server if update information retrieved correctly otherwise false
	 */
	public function getUpdates(Http $httpClient) {
		// Check if updates checker is disabled
		if($this->getComponentParams()->get('disable_version_checker', 0)) {
			return false;
		}
		
		// Updates server remote URI
		$option = $this->getState ( 'option', 'com_gdpr' );
		if (! $option) {
			return false;
		}
		$url = SERVER_REMOTE_URI . $option . UPDATES_FORMAT;
		
		// Try to get informations
		try {
			$response = $httpClient->get ( $url )->body;
			if ($response) {
				$decodedUpdateInfos = json_decode ( $response );
			}
			return $decodedUpdateInfos;
		} catch ( GdprException $e ) {
			return false;
		} catch ( \Exception $e ) {
			return false;
		}
	}
	
	/**
	 * Class constructor
	 * 
	 * @access public
	 * @param array $config        	
	 * @return Object&
	 */
	public function __construct($config = array(), MVCFactoryInterface $factory = null) {
		// Parent constructor
		parent::__construct ( $config, $factory );
	}
}