<?php
namespace JExtstore\Component\Gdpr\Administrator\Framework\Helpers;
/**
 * @package GDPR::FRAMEWORK::administrator::components::com_gdpr
 * @subpackage framework
 * @subpackage helpers
 * @author Joomla! Extensions Store
 * @copyright (C) 2015 - Joomla! Extensions Store
 * @license GNU/GPLv2 http://www.gnu.org/licenses/gpl-2.0.html
 */
defined('_JEXEC') or die('Restricted access');

// CSV import fields
define ('COM_GDPR_COOKIE', 0);
define ('COM_GDPR_CATEGORY', 1);
define ('COM_GDPR_DESCRIPTIONHTML', 2);
define ('COM_GDPR_COOKIE_EXPIRATION', 3);
define ('COM_GDPR_COOKIE_ALIAS', 4);
define ('COM_GDPR_PUBLISHED', 5);
define ('COM_GDPR_FIELDS_NUM', 6);

use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\String\StringHelper;
use Joomla\Event\Event;
use JExtstore\Component\Gdpr\Administrator\Framework\Exception as GdprException;

/**
 * Manager of the import/export configuration and migration of settings JSON encoded
 * 
 * @package GDPR::FRAMEWORK::administrator::components::com_gdpr
 * @subpackage framework
 * @subpackage file
 * @since 4.5
 */
class Config extends Error {
	/**
	 * Database connector
	 * 
	 * @access private
	 * @var Object
	 */
	private $dbo;
	
	/**
	 * Application object
	 *
	 * @access private
	 * @var Object
	 */
	private $app;
	
	/**
	 * Clean the cache
	 * @param   string   $group      The cache group
	 * @param   integer  $client_id  The ID of the client
	 * @return  void
	 * @since   11.1
	 */
	private function cleanComponentCache($group = null, $client_id = 0) {
		// Initialise variables;
		$conf = $this->app->getConfig();
		
		$options = array(
				'defaultgroup' => ($group) ? $group : $this->app->getInput()->get('option'),
				'cachebase' => ($client_id) ? JPATH_ADMINISTRATOR . '/cache' : $conf->get('cache_path', JPATH_CACHE));
		
		$cache = Factory::getContainer()->get(\Joomla\CMS\Cache\CacheControllerFactoryInterface::class)->createCacheController( 'callback', $options );
		$cache->clean();
		
		// Trigger the onContentCleanCache event.
		$this->app->getDispatcher()->dispatch('onContentCleanCache', new Event('onContentCleanCache', $options));
		
	}
	
	/**
	 * Store uploaded file to cache folder,
	 * fully manage error messages and ask for database insert
	 * 
	 * @access public
	 * @return boolean 
	 */
	public function import() {
		// Get file info
		$file = $this->app->getInput()->files->get('configurationimport', null, 'raw');
		$tmpFile = $file['tmp_name'];
		$tmpFileName = $file['name'];
		try {
			if(!$tmpFile || !$tmpFileName) {
				throw new GdprException(Text::_('COM_GDPR_NOFILE_SELECTED'), 'error');
			}
			
			$tmpFileSize = $file['size'];
			$allowedFileSize = 2 * 1024 * 1024; // MB->Bytes
			if($tmpFileSize > $allowedFileSize) {
				throw new GdprException(Text::_('COM_GDPR_SIZE_ERROR') .' Max 2MB.', 'error');
			}
			
			$tmpFileExtension = @array_pop(explode('.', $tmpFileName));
			if($tmpFileExtension != 'json') {
				throw new GdprException(Text::_('COM_GDPR_EXT_ERROR'), 'error');
			}

			// Deserialize contents
			$fileContents = file_get_contents($tmpFile);
			if($fileContents) {
				$objectToRestore = json_decode($fileContents);
			}
			
			if(!is_object($objectToRestore)) {
				throw new GdprException(Text::_('COM_GDPR_INVALID_IMPORT_DATA'), 'error');
			}
			
			$queryImport = 	"UPDATE " . $this->dbo->quoteName('#__extensions') .
						   	"\n SET " . $this->dbo->quoteName('params') . " = " . $this->dbo->quote($fileContents) .
						   	"\n WHERE " . $this->dbo->quoteName('type') . " = " . $this->dbo->quote('component') .
							"\n AND " . $this->dbo->quoteName('element') . " = " . $this->dbo->quote('com_gdpr') .
							"\n AND " . $this->dbo->quoteName('client_id') . " = 1";
			$this->dbo->setQuery($queryImport);
			$this->dbo->execute();
			
			// Clean the cache.
			$this->cleanComponentCache('_system', 0);
			$this->cleanComponentCache('_system', 1);
		}
		catch(GdprException $e) {
			$this->setError($e);
			return false;
		} catch (\Exception $e) {
			$gdprException = new GdprException($e->getMessage(), 'error');
			$this->setError($gdprException);
			return false;
		}
		
		return true;
	}

	/**
	 * Download uploaded file message
	 * 
	 * @access public
	 * @return boolean
	 */
	public function export($ids = null) { 
		// Load data from DB 
		try {
			$query = "SELECT" . $this->dbo->quoteName('params') .
					 "\n FROM " . $this->dbo->quoteName('#__extensions') .
					 "\n WHERE " . $this->dbo->quoteName('type') . " = " . $this->dbo->quote('component') .
					 "\n AND " . $this->dbo->quoteName('element') . " = " . $this->dbo->quote('com_gdpr') .
					 "\n AND " . $this->dbo->quoteName('client_id') . " = 1";
			$this->dbo->setQuery($query);
			$resultInfo = $this->dbo->loadResult();
			if(!$resultInfo) {
				throw new GdprException(Text::_('COM_GDPR_ERROR_NODATA_TOEXPORT'), 'error');
			}
		} catch(GdprException $e) {
			$this->setError($e);
			return false;
		} catch (\Exception $e) {
			$gdprException = new GdprException($e->getMessage(), 'error');
			$this->setError($gdprException);
			return false;
		}
		
		$fsize = StringHelper::strlen($resultInfo);
		$cont_dis = 'attachment';
		$mimeType = 'application/json';
		
		// required for IE, otherwise Content-disposition is ignored
		if (ini_get ( 'zlib.output_compression' )) {
			ini_set ( 'zlib.output_compression', 'Off' );
		}
		header ( "Pragma: public" );
		header ( "Cache-Control: must-revalidate, post-check=0, pre-check=0" );
		header ( "Expires: 0" );
		header ( "Content-Transfer-Encoding: binary" );
		header ( 'Content-Disposition:' . $cont_dis . ';' . ' filename="gdpr_configuration.json";' . ' size=' . $fsize . ';' ); //RFC2183
		header ( "Content-Type: " . $mimeType ); // MIME type
		header ( "Content-Length: " . $fsize );
		if (! ini_get ( 'safe_mode' )) { // set_time_limit doesn't work in safe mode
			@set_time_limit ( 0 );
		}
		// No encoding - we aren't using compression... (RFC1945)
		//header("Content-Encoding: none");
		//header("Vary: none");
		echo $resultInfo;
		
		exit();
	}

	/**
	 * Store uploaded file to cache folder,
	 * fully manage error messages and ask for database insert
	 *
	 * @access public
	 * @param Object $componentConfig
	 * @param boolean $defaultFile
	 * @return boolean
	 */
	public function importCookieDescriptions($componentConfig, $defaultFile = false) {
		// Setup the default file import
		if($defaultFile) {
			$tmpFile = JPATH_ROOT . '/administrator/components/com_gdpr/Framework/Helpers/cookie_descriptions.csv';
			$tmpFileName = 'cookie_descriptions.csv';
		} else {
			// Get file info
			$file = $this->app->getInput()->files->get('configurationimport', null, 'raw');
			$tmpFile = $file['tmp_name'];
			$tmpFileName = $file['name'];
		}
		
		$delimiter = $componentConfig->get('csv_delimiter', ';');
		$enclosure = $componentConfig->get('csv_enclosure', '"');
		
		try {
			if(!$tmpFile || !$tmpFileName) {
				throw new GdprException(Text::_('COM_GDPR_NOFILE_SELECTED'), 'error');
			}
			
			$tmpFileExtension = @array_pop(explode('.', $tmpFileName));
			if($tmpFileExtension != 'csv') {
				throw new GdprException(Text::_('COM_GDPR_COOKIEDESCRIPTIONS_EXT_ERROR'), 'error');
			}
			
			// Deserialize contents
			$fileHandle = fopen($tmpFile, "r");
			if(!is_resource($fileHandle)) {
				throw new GdprException(Text::_('COM_GDPR_DATA_FILE_NOT_READABLE'), 'error');
			}
			
			// Parse the CSV files dataset into an importable array
			$skip = true;
			$dbQueryArray = array();
			while ( $csvRecord = fgetcsv ( $fileHandle, 0, $delimiter, $enclosure ) ) {
				// Skip prima riga intestazioni
				if($skip) {
					$skip = false;
					continue;
				}
				//Insert
				array_push ( $dbQueryArray, $csvRecord );
			}
			
			// Check if some valid data to import are available
			if(!count($dbQueryArray)) {
				throw new GdprException(Text::_('COM_GDPR_NO_IMPORT_DATA_FOUND'), 'warning');
			}
			
			// Prepare the values array
			foreach ($dbQueryArray as $dbRecord) {
				// Ensure at least that the number of csv fields are correct
				if(count($dbRecord) != COM_GDPR_FIELDS_NUM) {
					continue;
				}
				
				// Check if the link as primary key already exists in this table
				$selectQuery = 	"SELECT" . $this->dbo->quoteName('id') .
								"\n FROM " . $this->dbo->quoteName('#__gdpr_cookie_descriptions') .
								"\n WHERE" .
								$this->dbo->quoteName('cookie') . " = " . $this->dbo->quote($dbRecord[COM_GDPR_COOKIE]) .
								"\n AND " . $this->dbo->quoteName('category') . " = " . $this->dbo->quote($dbRecord[COM_GDPR_CATEGORY]);
				$recordExists = $this->dbo->setQuery ( $selectQuery )->loadResult();

				// Sanitize the imported HTML field to avoid cross site scripting
				$dbRecord[COM_GDPR_DESCRIPTIONHTML] = preg_replace('#<script(.*?)>(.*?)</script>#is', '$2', $dbRecord[COM_GDPR_DESCRIPTIONHTML]);
				
				// If the link exists just update it, otherwise insert a new one
				if($recordExists) {
					$query = "UPDATE" .
							 "\n " . $this->dbo->quoteName('#__gdpr_cookie_descriptions') .
							 "\n SET " .
							 "\n " . $this->dbo->quoteName('cookie') . " = " . $this->dbo->quote($dbRecord[COM_GDPR_COOKIE]) . "," .
							 "\n " . $this->dbo->quoteName('category') . " = " . $this->dbo->quote($dbRecord[COM_GDPR_CATEGORY]) . "," .
							 "\n " . $this->dbo->quoteName('descriptionhtml') . " = " . $this->dbo->quote($dbRecord[COM_GDPR_DESCRIPTIONHTML]) . "," .
							 "\n " . $this->dbo->quoteName('expiration') . " = " . $this->dbo->quote($dbRecord[COM_GDPR_COOKIE_EXPIRATION]) . "," .
							 "\n " . $this->dbo->quoteName('alias') . " = " . $this->dbo->quote($dbRecord[COM_GDPR_COOKIE_ALIAS]) . "," .
							 "\n " . $this->dbo->quoteName('published') . " = " . (int)($dbRecord[COM_GDPR_PUBLISHED]) .
							 "\n WHERE " .
							 "\n " . $this->dbo->quoteName('id') . " = " . (int)$recordExists;
					$this->dbo->setQuery ( $query );
				} else {
					$query = "INSERT INTO" .
							 "\n " . $this->dbo->quoteName('#__gdpr_cookie_descriptions') . "(" .
							 $this->dbo->quoteName('cookie') . "," .
							 $this->dbo->quoteName('category') . "," .
							 $this->dbo->quoteName('descriptionhtml') . "," .
							 $this->dbo->quoteName('expiration') . "," .
							 $this->dbo->quoteName('alias') . "," .
							 $this->dbo->quoteName('published') . ") VALUES (" .
							 $this->dbo->quote($dbRecord[COM_GDPR_COOKIE]) . "," .
							 $this->dbo->quote($dbRecord[COM_GDPR_CATEGORY]) . "," .
							 $this->dbo->quote($dbRecord[COM_GDPR_DESCRIPTIONHTML]) . "," .
							 $this->dbo->quote($dbRecord[COM_GDPR_COOKIE_EXPIRATION]) . "," .
							 $this->dbo->quote($dbRecord[COM_GDPR_COOKIE_ALIAS]) . "," .
							 (int)$dbRecord[COM_GDPR_PUBLISHED] . ")";
					$this->dbo->setQuery ( $query );
				}
				$this->dbo->execute ();
			}
		}
		catch(GdprException $e) {
			$this->setError($e);
			return false;
		} catch (\Exception $e) {
			$gdprException = new GdprException(Text::sprintf('COM_GDPR_ERROR_STORING_DATA', $e->getMessage()), 'error');
			$this->setError($gdprException);
			return false;
		}
		
		return true;
	}

	/**
	 * Store the partial json configuration for cookies options
	 *
	 * @access public
	 * @return boolean
	 */
	public function importPartial() {
		// Get file info
		$file = $this->app->getInput()->files->get('configurationimport', null, 'raw');
		$tmpFile = $file['tmp_name'];
		$tmpFileName = $file['name'];
		try {
			if(!$tmpFile || !$tmpFileName) {
				throw new GdprException(Text::_('COM_GDPR_NOFILE_SELECTED'), 'error');
			}
			
			$tmpFileSize = $file['size'];
			$allowedFileSize = 2 * 1024 * 1024; // MB->Bytes
			if($tmpFileSize > $allowedFileSize) {
				throw new GdprException(Text::_('COM_GDPR_SIZE_ERROR') .' Max 2MB.', 'error');
			}
			
			$tmpFileExtension = @array_pop(explode('.', $tmpFileName));
			if($tmpFileExtension != 'json') {
				throw new GdprException(Text::_('COM_GDPR_EXT_ERROR'), 'error');
			}
			
			// Deserialize contents
			$fileContents = file_get_contents($tmpFile);
			if($fileContents) {
				$objectToRestore = json_decode($fileContents, true);
			}
			
			if(!is_array($objectToRestore)) {
				throw new GdprException(Text::_('COM_GDPR_INVALID_IMPORT_DATA'), 'error');
			}
			
			// Load data from DB
			try {
				$query = "SELECT" . $this->dbo->quoteName('params') .
						 "\n FROM " . $this->dbo->quoteName('#__extensions') .
						 "\n WHERE " . $this->dbo->quoteName('type') . " = " . $this->dbo->quote('component') .
						 "\n AND " . $this->dbo->quoteName('element') . " = " . $this->dbo->quote('com_gdpr') .
						 "\n AND " . $this->dbo->quoteName('client_id') . " = 1";
				$this->dbo->setQuery($query);
				$resultInfo = $this->dbo->loadResult();
				if(!$resultInfo) {
					throw new GdprException(Text::_('COM_GDPR_ERROR_NODATA_TOEXPORT'), 'error');
				}
			} catch(GdprException $e) {
				$this->setError($e);
				return false;
			} catch (Exception $e) {
				$gdprException = new GdprException($e->getMessage(), 'error');
				$this->setError($gdprException);
				return false;
			}
			
			// Decode all params
			$fullParamsArray = json_decode($resultInfo, true);
			
			// Merge imported params
			$paramsToImport = json_encode(array_merge($fullParamsArray, $objectToRestore));
			
			$queryImport = 	"UPDATE " . $this->dbo->quoteName('#__extensions') .
							"\n SET " . $this->dbo->quoteName('params') . " = " . $this->dbo->quote($paramsToImport) .
							"\n WHERE " . $this->dbo->quoteName('type') . " = " . $this->dbo->quote('component') .
							"\n AND " . $this->dbo->quoteName('element') . " = " . $this->dbo->quote('com_gdpr') .
							"\n AND " . $this->dbo->quoteName('client_id') . " = 1";
			$this->dbo->setQuery($queryImport);
			$this->dbo->execute();
			
			// Clean the cache.
			$this->cleanComponentCache('_system', 0);
			$this->cleanComponentCache('_system', 1);
		}
		catch(GdprException $e) {
			$this->setError($e);
			return false;
		} catch (Exception $e) {
			$gdprException = new GdprException($e->getMessage(), 'error');
			$this->setError($gdprException);
			return false;
		}
		
		return true;
	}
	
	/**
	 * Export a partial config related only to the 2 cookies tabs
	 *
	 * @access public
	 * @return boolean
	 */
	public function exportPartial() {
		$app = Factory::getApplication();
		
		// list of subparams to export
		$paramsToExport = array (
				'enable_cookie_consent' => '',
				'cookie_consent_lifetime' => '',
				'cookie_consent_samesite_policy' => '',
				'cookie_consent_secure' => '',
				'enable_log_cookie_consent' => '',
				'compliance_type' => '',
				'disable_first_reload' => '',
				'block_joomla_session_cookie' => '',
				'block_external_cookies_domains' => '',
				'external_blocking_mode' => '',
				'external_advanced_blocking_mode_tags' => '',
				'external_advanced_blocking_mode_custom_attribute' => '',
				'external_cookies_domains' => '',
				'allow_local_cookies' => '',
				'block_cookie_define' => '',
				'block_local_cookies_server_side' => '',
				'auto_accept_on_next_page' => '',
				'revokable' => '',
				'lawbycountry' => '',
				'checkboxlawbycountry' => '',
				'cache_geolocation_country' => '',
				'country_enable_reload' => '',
				'custom_has_law_countries' => '',
				'country_accept_reload_timeout' => '',
				'usa_ccpa' => '',
				'kill_cookies_ajax_requests' => '',
				'open_always_declined' => '',
				'default_closed_toolbar' => '',
				'dismiss_onscroll' => '',
				'dismiss_ontimeout' => '',
				'container_selector' => '',
				'hide_on_mobile_devices' => '',
				'auto_floating_on_mobile' => '',
				'auto_floating_on_mobile_threshold' => '',
				'auto_redirect_on_decline' => '',
				'auto_redirect_on_decline_link' => '',
				'show_reload_message' => '',
				'show_reload_message_text' => '',
				'placeholder_blocked_resources' => '',
				'placeholder_blocked_resources_action' => '',
				'placeholder_blocked_resources_text' => '',
				'placeholder_text_color' => '',
				'placeholder_background_color' => '',
				'placeholder_background_image' => '',
				'placeholder_custom_width' => '',
				'placeholder_custom_height' => '',
				'placeholder_invididual_blocked_resources_text' => '',
				'placeholder_individual_blocked_resources_action' => '',
				'placeholder_onpage_unlock' => '',
				'scripts_onpage_unlock' => '',
				'auto_detect_youtube_poster' => '0',
				'scripts_onpage_unlock' => 'AIzaSyAV_WIyYrUkFV1H8OKFYG8wIK8wVH9c82U',
				'layout' => '',
				'theme' => '',
				'buttons_theme' => '',
				'position' => '',
				'center_theme' => '',
				'center_theme_swift' => '',
				'googlecmp_template' => '',
				'category_read_more' => '',
				'position_center_simple_backdrop' => '',
				'position_center_blur_effect' => '',
				'position_center_blur_effect_type' => '',
				'position_center_blur_effect_percentage' => '',
				'prevent_page_scrolling' => '',
				'switcher_style' => '',
				'positionment_type' => '',
				'revokeposition' => '',
				'revocabletheme' => '',
				'popup_effect' => '',
				'popup_background' => '',
				'popup_fontsize' => '',
				'revocable_button_fontsize' => '',
				'remove_decline_button' => '',
				'remove_dismiss_button' => '',
				'popup_padding' => '',
				'popup_border_radius' => '0',
				'popup_background_opacity' => '',
				'boxbar_maxwidth' => '',
				'popup_text' => '',
				'popup_link' => '',
				'button_background' => '',
				'button_border' => '',
				'button_text' => '',
				'highlight_background' => '',
				'highlight_border' => '',
				'highlight_text' => '',
				'highlight_dismiss_background' => '',
				'highlight_dismiss_border' => '',
				'highlight_dismiss_text' => '',
				'hide_revokable_button' => '',
				'hide_revokable_button_onscroll' => '',
				'custom_revokable_button' => '',
				'custom_revokable_button_action' => '',
				'header' => '',
				'message' => '',
				'dismiss_text' => '',
				'allow_text' => '',
				'deny_text' => '',
				'cookie_policy_link_text' => '',
				'cookie_policy_link' => '',
				'cookie_policy_revocable_tab_text' => '',
				'privacy_policy_link_text' => '',
				'privacy_policy_link' => '',
				'show_links' => '',
				'blank_links' => '',
				'auto_open_privacy_policy' => '',
				'deny_message_enabled' => '',
				'deny_message' => '',
				'use_fancybox_links' => '',
				'fancybox_width' => '',
				'fancybox_height' => '',
				'popup_format_template' => '',
				'use_cookie_policy_contents' => '',
				'cookie_policy_contents' => '',
				'use_privacy_policy_contents' => '',
				'privacy_policy_contents' => '',
				'enable_gdpr_bulk_consent' => '',
				'bulk_consent_domains' => '',
				'bulk_consent_api_transport' => '',
				'enable_custom_script_exec_generic' => '',
				'custom_script_exec_generic' => '',
				'cookie_settings_label' => '',
				'cookie_settings_desc' => '',
				'categories_checkbox_template' => '',
				'toggle_cookie_settings' => '',
				'toggle_cookie_settings_text' => '',
				'toggle_cookie_settings_button_background' => '',
				'toggle_cookie_settings_button_border' => '',
				'toggle_cookie_settings_button_text' => '',
				'categories_border_radius' => '',
				'propagate_categories_session' => '',
				'always_propagate_categories_session' => '',
				'always_reload_after_categories_change' => '',
				'preserve_locked_categories' => '',
				'decline_button_behavior' => '',
				'reload_onfirst_declineall' => '',
				'allowall_showbutton' => '',
				'allowall_text' => '',
				'allowall_button_background' => '',
				'allowall_button_timing_ajax' => '',
				'allowall_button_border' => '',
				'allowall_button_text' => '',
				'include_accept_button' => '',
				'track_consent_date' => '',
				'exec_custom_scripts_once' => '',
				'optout_individual_resources' => '',
				'block_individual_resources_serverside' => '',
				'optout_individual_resources_exclude_locked_categories' => '',
				'disable_switchers_optout_category' => '',
				'auto_accept_categories' => '',
				'allowall_individual_resources' => '',
				'categories_cookies_blocking_mode' => '',
				'block_local_storage' => '',
				'block_session_storage' => '',
				'block_external_categories_domains' => '',
				'cookie_category1_enable' => '',
				'cookie_category1_checked' => '',
				'cookie_category1_locked' => '',
				'cookie_category1_name' => '',
				'cookie_category1_description' => '',
				'cookie_category1_list' => '',
				'domains_category1_list' => '',
				'cookie_category1_include_list' => '',
				'domains_category1_include_list' => '',
				'services_category1_include_list' => '',
				'enable_custom_script_exec_category1' => '',
				'custom_script_exec_category1' => '',
				'enable_custom_decline_script_exec_category1' => '',
				'custom_script_decline_exec_category1' => '',
				'cookie_category2_enable' => '',
				'cookie_category2_checked' => '',
				'cookie_category2_locked' => '',
				'cookie_category2_name' => '',
				'cookie_category2_description' => '',
				'cookie_category2_list' => '',
				'domains_category2_list' => '',
				'cookie_category2_include_list' => '',
				'domains_category2_include_list' => '',
				'services_category2_include_list' => '',
				'enable_custom_script_exec_category2' => '',
				'custom_script_exec_category2' => '',
				'enable_custom_decline_script_exec_category2' => '',
				'custom_script_decline_exec_category2' => '',
				'cookie_category3_enable' => '',
				'cookie_category3_checked' => '',
				'cookie_category3_locked' => '',
				'cookie_category3_name' => '',
				'cookie_category3_description' => '',
				'cookie_category3_list' => '',
				'domains_category3_list' => '',
				'cookie_category3_include_list' => '',
				'domains_category3_include_list' => '',
				'services_category3_include_list' => '',
				'enable_custom_script_exec_category3' => '',
				'custom_script_exec_category3' => '',
				'enable_custom_decline_script_exec_category3' => '',
				'custom_script_decline_exec_category3' => '',
				'cookie_category4_enable' => '',
				'cookie_category4_checked' => '',
				'cookie_category4_locked' => '',
				'cookie_category4_name' => '',
				'cookie_category4_description' => '',
				'cookie_category4_list' => '',
				'domains_category4_list' => '',
				'cookie_category4_include_list' => '',
				'domains_category4_include_list' => '',
				'services_category4_include_list' => '',
				'enable_custom_script_exec_category4' => '',
				'custom_script_exec_category4' => '',
				'enable_custom_decline_script_exec_category4' => '',
				'custom_script_decline_exec_category4' => ''
		);
		
		// Load data from DB
		try {
			$query = "SELECT" . $this->dbo->quoteName('params') .
					 "\n FROM " . $this->dbo->quoteName('#__extensions') .
					 "\n WHERE " . $this->dbo->quoteName('type') . " = " . $this->dbo->quote('component') .
					 "\n AND " . $this->dbo->quoteName('element') . " = " . $this->dbo->quote('com_gdpr') .
					 "\n AND " . $this->dbo->quoteName('client_id') . " = 1";
			$this->dbo->setQuery($query);
			$resultInfo = $this->dbo->loadResult();
			if(!$resultInfo) {
				throw new GdprException(Text::_('COM_GDPR_NODATA_EXPORT'), 'error');
			}
		} catch(GdprException $e) {
			$this->setError($e);
			return false;
		} catch (Exception $e) {
			$gdprException = new GdprException($e->getMessage(), 'error');
			$this->setError($gdprException);
			return false;
		}
		
		// Decode all params
		$fullParamsArray = json_decode($resultInfo, true);
		foreach ( $paramsToExport as $paramName => $paramValue ) {
			$paramsToExport[$paramName] = $fullParamsArray[$paramName];
		}

		// Recover the services params
		for($c = 1, $m = 4; $c <= $m; $c ++) {
			$serviceParamName = 'services_category' . $c . '_list';
			$paramsToExport[$serviceParamName] = $app->getInput()->post->getString('params')[$serviceParamName];
		}
		
		$resultInfo = json_encode($paramsToExport);
		
		$fsize = StringHelper::strlen($resultInfo);
		$cont_dis = 'attachment';
		$mimeType = 'application/json';
		
		// required for IE, otherwise Content-disposition is ignored
		if (ini_get ( 'zlib.output_compression' )) {
			ini_set ( 'zlib.output_compression', 'Off' );
		}
		header ( "Pragma: public" );
		header ( "Cache-Control: must-revalidate, post-check=0, pre-check=0" );
		header ( "Expires: 0" );
		header ( "Content-Transfer-Encoding: binary" );
		header ( 'Content-Disposition:' . $cont_dis . ';' . ' filename="gdpr_cookie_configuration.json";' . ' size=' . $fsize . ';' ); //RFC2183
		header ( "Content-Type: " . $mimeType ); // MIME type
		header ( "Content-Length: " . $fsize );
		if (! ini_get ( 'safe_mode' )) { // set_time_limit doesn't work in safe mode
			@set_time_limit ( 0 );
		}
		// No encoding - we aren't using compression... (RFC1945)
		//header("Content-Encoding: none");
		//header("Vary: none");
		echo $resultInfo;
		
		exit();
	}

	/**
	 * Class constructor
	 * 
	 * @access public
	 * @param Object $dbo
	 * @param Object $app
	 * @return Object &
	 */
	public function __construct($dbo, $app) {
		// DB connector
		$this->dbo = $dbo;
		
		// Application
		$this->app = $app;
	}
}