<?php

/**
 * @copyright
 * @package    Easy Joomla Backup Pro - EJB for Joomla! 5
 * @author     Viktor Vogel <admin@kubik-rubik.de>
 * @version    5.1.0.0-PRO - 2024-07-28
 * @link       https://kubik-rubik.de/ejb-easy-joomla-backup
 *
 * @license    GNU/GPL
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

namespace KubikRubik\Component\EasyJoomlaBackup\Administrator\Controller;

defined('_JEXEC') || die('Restricted access');

use Exception;
use Joomla\CMS\MVC\Controller\BaseController;
use Joomla\CMS\{Factory, Language\Text, Router\Route, Session\Session};
use KubikRubik\Component\EasyJoomlaBackup\Administrator\Helper\EasyJoomlaBackupHelper;
use KubikRubik\Component\EasyJoomlaBackup\Administrator\Model\CreateModel;
use KubikRubik\Component\EasyJoomlaBackup\Administrator\Model\BackupsModel;
use RuntimeException;

use function defined;

/**
 * Class DisplayController
 *
 * @version 5.0.0.1-PRO
 * @since   5.0.0.0-PRO
 */
class DisplayController extends BaseController
{
    /**
     * @var string $default_view
     * @version 5.0.0.1-PRO
     * @since   5.0.0.0-PRO
     */
    protected $default_view = 'Backups';

    /**
     * Loads the full backup template
     *
     * @throws Exception
     * @version 5.0.0.1-PRO
     * @since   5.0.0.0-PRO
     */
    public function fullbackup(): void
    {
        if (!Factory::getApplication()->getIdentity()->authorise('easyjoomlabackup.fullbackup', 'com_easyjoomlabackup')) {
            throw new RuntimeException(Text::_('JERROR_ALERTNOAUTHOR'), 404);
        }

        $this->isBackupStorageLocationWritableCheck();

        $this->input = Factory::getApplication()->input;
        $this->input->set('view', 'Create');
        $this->input->set('hidemainmenu', 1);

        $this->display();
    }

    /**
     * Is the backup storage location writable by PHP
     *
     * @return void
     * @since 5.0.0.0-PRO
     */
    private function isBackupStorageLocationWritableCheck(): void
    {
        if (!EasyJoomlaBackupHelper::isBackupStorageLocationWritable()) {
            $msg = Text::_('COM_EASYJOOMLABACKUP_BACKUPSTORAGELOCATION_WRITABLE_NOT_ERROR');
            $type = 'warning';
            $this->setRedirect(Route::_('index.php?option=com_easyjoomlabackup', false), $msg, $type);
        }
    }

    /**
     * Loads the database backup template
     *
     * @throws Exception
     * @version 5.0.0.1-PRO
     * @since   5.0.0.0-PRO
     */
    public function databasebackup(): void
    {
        if (!Factory::getApplication()->getIdentity()->authorise('easyjoomlabackup.databasebackup', 'com_easyjoomlabackup')) {
            throw new RuntimeException(Text::_('JERROR_ALERTNOAUTHOR'), 404);
        }

        $this->isBackupStorageLocationWritableCheck();

        $this->input = Factory::getApplication()->input;
        $this->input->set('view', 'Create');
        $this->input->set('hidemainmenu', 1);

        $this->display();
    }

    /**
     * Loads the file backup template
     *
     * @throws Exception
     * @version 5.0.0.1-PRO
     * @since   5.0.0.0-PRO
     */
    public function filebackup(): void
    {
        if (!Factory::getApplication()->getIdentity()->authorise('easyjoomlabackup.filebackup', 'com_easyjoomlabackup')) {
            throw new RuntimeException(Text::_('JERROR_ALERTNOAUTHOR'), 404);
        }

        $this->isBackupStorageLocationWritableCheck();

        $this->input = Factory::getApplication()->input;
        $this->input->set('view', 'Create');
        $this->input->set('hidemainmenu', 1);

        $this->display();
    }

    /**
     * Discovers backup files without database entries or database entries without corresponding backup archives
     *
     * @throws Exception
     * @version 5.0.0.1-PRO
     * @since   5.0.0.0-PRO
     */
    public function discover(): void
    {
        Session::checkToken() || jexit('Invalid Token');

        if (!Factory::getApplication()->getIdentity()->authorise('easyjoomlabackup.discover', 'com_easyjoomlabackup')) {
            throw new RuntimeException(Text::_('JERROR_ALERTNOAUTHOR'), 404);
        }

        $model = new BackupsModel();

        $msg = Text::_('COM_EASYJOOMLABACKUP_DISCOVER_SUCCESS');
        $type = 'message';

        if (!$model->discover()) {
            $msg = Text::_('COM_EASYJOOMLABACKUP_DISCOVER_NOTICE');
            $type = 'notice';
        }

        $this->setRedirect(Route::_('index.php?option=com_easyjoomlabackup', false), $msg, $type);
    }

    /**
     * Deletes selected entries and the corresponding backup archives
     *
     * @throws Exception
     * @version 5.0.0.1-PRO
     * @since   5.0.0.0-PRO
     */
    public function remove(): void
    {
        Session::checkToken() || jexit('Invalid Token');

        if (!Factory::getApplication()->getIdentity()->authorise('core.delete', 'com_easyjoomlabackup')) {
            throw new RuntimeException(Text::_('JERROR_ALERTNOAUTHOR'), 404);
        }

        $model = new CreateModel();

        $msg = Text::_('COM_EASYJOOMLABACKUP_BACKUP_DELETED');
        $type = 'message';

        if (!$model->delete()) {
            $msg = Text::_('COM_EASYJOOMLABACKUP_BACKUP_DELETED_ERROR');
            $type = 'error';
        }

        $this->setRedirect(Route::_('index.php?option=com_easyjoomlabackup', false), $msg, $type);
    }
}
