<?php

/**
 * @copyright
 * @package    Easy Joomla Backup Pro - EJB for Joomla! 5
 * @author     Viktor Vogel <admin@kubik-rubik.de>
 * @version    5.1.0.0-PRO - 2024-07-28
 * @link       https://kubik-rubik.de/ejb-easy-joomla-backup
 *
 * @license    GNU/GPL
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

namespace KubikRubik\Component\EasyJoomlaBackup\Administrator\Controller;

defined('_JEXEC') || die('Restricted access');

use Exception;
use Joomla\CMS\MVC\Controller\BaseController;
use Joomla\CMS\{Factory, Language\Text, Router\Route, Session\Session, Response\JsonResponse};
use Joomla\Input\Input;
use KubikRubik\Component\EasyJoomlaBackup\Administrator\Helper\EasyJoomlaBackupHelper;
use KubikRubik\Component\EasyJoomlaBackup\Administrator\Model\CreateModel;
use RuntimeException;

use function defined;

/**
 * Class CreateController
 *
 * @version 5.0.0.1-PRO
 * @since   5.0.0.0-PRO
 */
class CreateController extends BaseController
{
    /**
     * @var Input $input
     * @since 5.0.0.0-PRO
     */
    protected $input;

    /**
     * Aborts the selected backup process
     *
     * @since 5.0.0.0-PRO
     */
    public function cancel(): void
    {
        $msg = Text::_('COM_EASYJOOMLABACKUP_BACKUP_CANCELLED');
        $this->setRedirect('index.php?option=com_easyjoomlabackup', $msg, 'notice');
    }

    /**
     * Starts the full backup process with an ACL check
     *
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    public function backupCreateFullbackup(): void
    {
        if (!Factory::getApplication()->getIdentity()->authorise('easyjoomlabackup.' . EasyJoomlaBackupHelper::BACKUP_TYPE_FULL, 'com_easyjoomlabackup')) {
            throw new RuntimeException(Text::_('JERROR_ALERTNOAUTHOR'), 404);
        }

        $this->backupCreate(EasyJoomlaBackupHelper::BACKUP_TYPE_FULL);
    }

    /**
     * Creates the backup archive in dependence on the submitted type
     *
     * @param string $backupType
     *
     * @throws Exception
     * @version 5.0.0.1-PRO
     * @since   5.0.0.0-PRO
     */
    private function backupCreate(string $backupType): void
    {
        Session::checkToken('get') || jexit('Invalid Token');

        EasyJoomlaBackupHelper::increaseServerLimits();

        $model = new CreateModel();
        $hash = $this->input->get('hash', '');
        $password = $this->input->get('password', '');

        try {
            $result = $model->createBackupAjax($backupType, $hash, $password);

            if ($result['finished']) {
                EasyJoomlaBackupHelper::addMessage(Text::_('COM_EASYJOOMLABACKUP_BACKUP_SAVED'));

                if ($model->dropboxUploadAutomatic()) {
                    EasyJoomlaBackupHelper::addMessage(Text::_('COM_EASYJOOMLABACKUP_DROPBOXUPLOADAUTOMATIC_SUCCESS'));
                }

                if ($model->removeBackupFilesMax()) {
                    EasyJoomlaBackupHelper::addMessage(Text::_('COM_EASYJOOMLABACKUP_MAXNUMBERBACKUPS_REMOVED'), EasyJoomlaBackupHelper::MESSAGE_TYPE_NOTICE);
                }
            }

            echo new JsonResponse($result);
        } catch (Exception $e) {
            EasyJoomlaBackupHelper::addMessage(Text::sprintf('COM_EASYJOOMLABACKUP_BACKUP_SAVED_ERROR_EXCEPTION', $e->getMessage()), EasyJoomlaBackupHelper::MESSAGE_TYPE_ERROR);
            EasyJoomlaBackupHelper::log($e->getMessage());

            echo new JsonResponse(null, '', true);
        }
    }

    /**
     * Starts the database backup process with an ACL check
     *
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    public function backupCreateDatabasebackup(): void
    {
        if (!Factory::getApplication()->getIdentity()->authorise('easyjoomlabackup.' . EasyJoomlaBackupHelper::BACKUP_TYPE_DATABASE, 'com_easyjoomlabackup')) {
            throw new RuntimeException(Text::_('JERROR_ALERTNOAUTHOR'), 404);
        }

        $this->backupCreate(EasyJoomlaBackupHelper::BACKUP_TYPE_DATABASE);
    }

    /**
     * Starts the file backup process with an ACL check
     *
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    public function backupCreateFilebackup(): void
    {
        if (!Factory::getApplication()->getIdentity()->authorise('easyjoomlabackup.' . EasyJoomlaBackupHelper::BACKUP_TYPE_FILE, 'com_easyjoomlabackup')) {
            throw new RuntimeException(Text::_('JERROR_ALERTNOAUTHOR'), 404);
        }

        $this->backupCreate(EasyJoomlaBackupHelper::BACKUP_TYPE_FILE);
    }

    /**
     * Calls the download screen for the selected backup entry
     *
     * @throws Exception
     * @version 5.0.0.1-PRO
     * @since   5.0.0.0-PRO
     */
    public function download(): void
    {
        if (!Factory::getApplication()->getIdentity()->authorise('easyjoomlabackup.download', 'com_easyjoomlabackup')) {
            throw new RuntimeException(Text::_('JERROR_ALERTNOAUTHOR'), 404);
        }

        $model = new CreateModel();

        if (!$model->download()) {
            $msg = Text::_('COM_EASYJOOMLABACKUP_DOWNLOAD_ERROR');
            $type = 'error';
            $this->setRedirect(Route::_('index.php?option=com_easyjoomlabackup', false), $msg, $type);
        }
    }

    /**
     * Uploads the selected backup archive to Dropbox
     *
     * @throws Exception
     * @version 5.0.0.1-PRO
     * @since   5.0.0.0-PRO
     */
    public function dropboxUpload(): void
    {
        if (!Factory::getApplication()->getIdentity()->authorise('easyjoomlabackup.dropboxUpload', 'com_easyjoomlabackup')) {
            throw new RuntimeException(Text::_('JERROR_ALERTNOAUTHOR'), 404);
        }

        EasyJoomlaBackupHelper::increaseServerLimits();

        $model = new CreateModel();

        $result = [
            'redirectUrl' => 'index.php?option=com_easyjoomlabackup',
        ];

        try {
            if ($model->dropboxUpload()) {
                EasyJoomlaBackupHelper::addMessage(Text::_('COM_EASYJOOMLABACKUP_UPLOAD_SUCCESS'));
                EasyJoomlaBackupHelper::resetDropboxUploadStatus();

                echo new JsonResponse($result);

                return;
            }

            EasyJoomlaBackupHelper::addMessage(Text::_('COM_EASYJOOMLABACKUP_UPLOAD_WARNING'), EasyJoomlaBackupHelper::MESSAGE_TYPE_WARNING);

            echo new JsonResponse($result);
        } catch (Exception $e) {
            EasyJoomlaBackupHelper::addMessage(Text::_('COM_EASYJOOMLABACKUP_UPLOAD_ERROR'), EasyJoomlaBackupHelper::MESSAGE_TYPE_ERROR);
            EasyJoomlaBackupHelper::log($e->getMessage());

            echo new JsonResponse($result);
        }
    }

    /**
     * Uploads the selected backup archive to a server using SFTP
     *
     * @throws Exception
     * @version 5.0.0.1-PRO
     * @since   5.0.0.0-PRO
     */
    public function sftpUpload(): void
    {
        if (!Factory::getApplication()->getIdentity()->authorise('easyjoomlabackup.sftpUpload', 'com_easyjoomlabackup')) {
            throw new RuntimeException(Text::_('JERROR_ALERTNOAUTHOR'), 404);
        }

        EasyJoomlaBackupHelper::increaseServerLimits();

        $model = new CreateModel();

        $result = [
            'redirectUrl' => 'index.php?option=com_easyjoomlabackup',
        ];

        try {
            if ($model->sftpUpload()) {
                EasyJoomlaBackupHelper::addMessage(Text::_('COM_EASYJOOMLABACKUP_UPLOAD_SUCCESS'));
                EasyJoomlaBackupHelper::resetSftpUploadStatus();

                echo new JsonResponse($result);

                return;
            }

            EasyJoomlaBackupHelper::addMessage(Text::_('COM_EASYJOOMLABACKUP_UPLOAD_WARNING'), EasyJoomlaBackupHelper::MESSAGE_TYPE_WARNING);

            echo new JsonResponse($result);
        } catch (Exception $e) {
            EasyJoomlaBackupHelper::addMessage(Text::_('COM_EASYJOOMLABACKUP_UPLOAD_ERROR'), EasyJoomlaBackupHelper::MESSAGE_TYPE_ERROR);
            EasyJoomlaBackupHelper::log($e->getMessage());

            echo new JsonResponse($result);
        }
    }
}
