<?php

/**
 * @copyright
 * @package    Easy Joomla Backup Pro - EJB for Joomla! 5.x
 * @author     Viktor Vogel <admin@kubik-rubik.de>
 * @version    5.0.1.0-PRO - 2024-01-29
 * @link       https://kubik-rubik.de/ejb-easy-joomla-backup
 *
 * @license    GNU/GPL
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

namespace KubikRubik\Component\EasyJoomlaBackup\Administrator\Helper;

defined('_JEXEC') || die('Restricted access');

use Exception;

use function defined;
use function is_array;

/**
 * Class DropboxHelper
 *
 * @package EasyJoomlaBackup
 * @version 5.0.1.0-PRO
 * @since   5.0.0.0-PRO
 */
class DropboxHelper
{
    /**
     * @var string $accessToken
     * @since 5.0.0.0-PRO
     */
    protected string $accessToken;

    /**
     * Dropbox constructor
     *
     * @param string $accessToken
     *
     * @since 5.0.0.0-PRO
     */
    public function __construct(string $accessToken)
    {
        $this->accessToken = trim($accessToken);
    }

    /**
     * Uploads a file to the specified folder
     *
     * @param string $targetPath
     * @param string $fileData
     *
     * @return bool
     * @throws Exception
     * @version 5.0.1.0-PRO
     * @since   5.0.0.0-PRO
     */
    public function upload(string $targetPath, string $fileData): bool
    {
        if (is_file($fileData)) {
            $fileSize = filesize($fileData);

            if ($fileSize > EasyJoomlaBackupHelper::UPLOAD_LIMIT_DROPBOX_BYTES) {
                return $this->uploadSession($targetPath, $fileData, $fileSize);
            }

            $fileData = file_get_contents($fileData);
        }

        $apiEndpoint = 'https://content.dropboxapi.com/2/files/upload';
        $headers = [
            'Content-Type: application/octet-stream',
            'Dropbox-API-Arg: {"path": "' . $targetPath . '", "mode": "add", "autorename": true}',
        ];

        $result = $this->request($apiEndpoint, $headers, $fileData);

        if (isset($result['error'])) {
            return false;
        }

        return true;
    }

    /**
     * Uploads a big file to the specified folder
     *
     * @param string $targetPath
     * @param string $fileData
     * @param int    $fileSize
     *
     * @return bool
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    private function uploadSession(string $targetPath, string $fileData, int $fileSize): bool
    {
        $count = 1;
        $sessionId = '';
        $handle = fopen($fileData, 'rb');
        $chunkSize = EasyJoomlaBackupHelper::UPLOAD_SESSION_CHUNK_BYTES;

        while (!feof($handle)) {
            $buffer = fread($handle, $chunkSize);

            if (empty($sessionId)) {
                $sessionId = $this->uploadSessionStart($buffer);

                ob_flush();
                flush();

                continue;
            }

            $offset = $count * $chunkSize;
            $this->uploadSessionAppend($buffer, $sessionId, $offset);

            ob_flush();
            flush();

            $count++;
        }

        fclose($handle);

        return $this->uploadSessionFinish('', $targetPath, $sessionId, $fileSize);
    }

    /**
     * Uploads a big file to the specified folder
     *
     * @param string $fileData
     *
     * @return string
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    private function uploadSessionStart(string $fileData): string
    {
        $apiEndpoint = 'https://content.dropboxapi.com/2/files/upload_session/start';
        $headers = [
            'Content-Type: application/octet-stream',
            'Dropbox-API-Arg: {"close": false}',
        ];

        $result = $this->request($apiEndpoint, $headers, $fileData);

        if (is_array($result) && isset($result['session_id'])) {
            return $result['session_id'];
        }

        return '';
    }

    /**
     * Main function for handling post requests
     *
     * @param string $apiEndpoint
     * @param array  $headers
     * @param string $data
     *
     * @return bool|mixed|string
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    public function request(string $apiEndpoint, array $headers, string $data): mixed
    {
        $curlHandle = curl_init($apiEndpoint);
        $headers[] = 'Authorization: Bearer ' . $this->accessToken;
        curl_setopt($curlHandle, CURLOPT_POST, true);
        curl_setopt($curlHandle, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($curlHandle, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($curlHandle, CURLOPT_POSTFIELDS, $data);
        $result = curl_exec($curlHandle);
        curl_close($curlHandle);

        return json_decode($result, true, 512, JSON_THROW_ON_ERROR);
    }

    /**
     * Uploads a big file to the specified folder
     *
     * @param string $fileData
     * @param string $sessionId
     * @param int    $offset
     *
     * @return bool
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    private function uploadSessionAppend(string $fileData, string $sessionId, int $offset): bool
    {
        $apiEndpoint = 'https://content.dropboxapi.com/2/files/upload_session/append_v2';
        $headers = [
            'Content-Type: application/octet-stream',
            'Dropbox-API-Arg: {"cursor": {"session_id": "' . $sessionId . '", "offset": ' . $offset . '}}',
        ];

        $result = $this->request($apiEndpoint, $headers, $fileData);

        return !(is_array($result) && isset($result['error']));
    }

    /**
     * Uploads a big file to the specified folder
     *
     * @param string $fileData
     * @param string $sessionId
     * @param string $targetPath
     * @param int    $offset
     *
     * @return bool
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    private function uploadSessionFinish(string $fileData, string $targetPath, string $sessionId, int $offset): bool
    {
        $apiEndpoint = 'https://content.dropboxapi.com/2/files/upload_session/finish';
        $headers = [
            'Content-Type: application/octet-stream',
            'Dropbox-API-Arg: {"cursor": {"session_id": "' . $sessionId . '", "offset": ' . $offset . '}, "commit": {"path": "' . $targetPath . '", "mode": "add", "autorename": true}}',
        ];

        $result = $this->request($apiEndpoint, $headers, $fileData);

        return !(is_array($result) && isset($result['error']));
    }

    /**
     * Lists all archives from the associated Dropbox folder
     *
     * @param string $path
     *
     * @return array
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    public function listFolder(string $path = ''): array
    {
        $apiEndpoint = 'https://api.dropboxapi.com/2/files/list_folder';
        $headers = [
            'Content-Type: application/json',
        ];
        $data = json_encode(['path' => $path, 'recursive' => true], JSON_THROW_ON_ERROR);

        $result = $this->request($apiEndpoint, $headers, $data);

        if (!is_array($result)) {
            return [];
        }

        return $result['entries'] ?? [];
    }
}
