<?php

/**
 * @copyright
 * @package     Easy Joomla Backup Pro - EJB for Joomla! 5.x
 * @author      Viktor Vogel <admin@kubik-rubik.de>
 * @version     5.0.1.0-PRO - 2024-01-29
 * @link        https://kubik-rubik.de/ejb-easy-joomla-backup
 *
 * @license     GNU/GPL
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

namespace KubikRubik\Plugin\System\EasyJoomlaBackupCronjob\Field;

defined('_JEXEC') || die('Restricted access');

use Exception;
use Joomla\CMS\{Filesystem\File, Form\FormField, Factory, Http\HttpFactory, Language\Text};
use Joomla\Registry\Registry;
use JsonException;
use RuntimeException;

use function defined;

/**
 * Pro security token validation field for Kubik-Rubik Joomla! extensions
 *
 * @since 5.0.0.0-PRO
 */
class ProSecurityTokenField extends FormField
{
    /**
     * @since 5.0.0.0-PRO
     */
    private const TOKEN_FILENAME = 'ProSecurityToken.php';

    /**
     * @var string $type
     * @since 5.0.0.0-PRO
     */
    protected $type = 'proSecurityToken';

    /**
     * @var string $parentclass
     * @since 5.0.0.0-PRO
     */
    protected $parentclass = 'hidden';

    /**
     * @var string $tokenPath
     * @since 5.0.0.0-PRO
     */
    protected string $tokenPath = '';

    /**
     * @throws JsonException|Exception
     * @since 5.0.0.0-PRO
     */
    protected function getInput(): string
    {
        $session = Factory::getApplication()->getSession();
        $this->tokenPath = JPATH_ROOT . '/' . $this->getAttribute('tokenPath') . '/' . self::TOKEN_FILENAME;

        if (!File::exists($this->tokenPath)) {
            $this->setProInputErrorOutput(Text::_('PRO_SECURITY_TOKEN_FILE_MISSING'), 'missingSecurityTokenFile');

            return '';
        }

        $proSecurityToken = $this->getProSecurityToken();

        if ($proSecurityToken === '') {
            $this->setProInputErrorOutput(Text::_('PRO_SECURITY_TOKEN_MISSING'), 'missingSecurityToken');

            return '';
        }

        $proSecurityTokenStatus = $session->get('proSecurityTokenStatus', null, 'proSecurityToken');
        $proSecurityTokenFileChecksum = $session->get('proSecurityTokenFileChecksum', null, 'proSecurityToken');

        if ($proSecurityTokenStatus === null || $proSecurityTokenFileChecksum === null || $proSecurityTokenFileChecksum !== md5_file($this->tokenPath)) {
            try {
                $proSecurityTokenStatus = $this->getProSecurityTokenStatus($proSecurityToken);
            } catch (Exception $e) {
                $errorMessage = $e->getMessage();

                if ($e instanceof JsonException) {
                    $errorMessage = Text::_('PRO_SECURITY_TOKEN_EXCEPTION_JSON_PARSE');
                }

                $this->setProInputErrorOutput($errorMessage, 'exceptionSecurityToken', 'warning');

                return '';
            }

            $session->set('proSecurityTokenStatus', $proSecurityTokenStatus, 'proSecurityToken');
            $session->set('proSecurityTokenFileChecksum', md5_file($this->tokenPath), 'proSecurityToken');
        }

        if (!$proSecurityTokenStatus) {
            $this->setProInputErrorOutput(Text::_('PRO_SECURITY_TOKEN_INVALID'), 'invalidSecurityToken');

            return '';
        }

        return '';
    }

    /**
     * @param string $message
     * @param string $linkId
     * @param string $type
     *
     * @return void
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    private function setProInputErrorOutput(string $message, string $linkId, string $type = 'error'): void
    {
        $this->appendProValidationButton('<a href="' . $this->getAttribute('downloadLink') . '?' . $linkId . '" style="color: inherit;">' . Text::_('PRO_SECURITY_DOWNLOAD_BUTTON') . '</a>');

        Factory::getApplication()->enqueueMessage($message, $type);
    }

    /**
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    private function appendProValidationButton(string $linkText): void
    {
        static $loaded = null;

        if ($loaded !== null) {
            return;
        }

        $proButton = '<joomla-toolbar-button class="btn-wrapper" id="toolbar-pro-validation"><button type="button" class="btn btn-info"><span class="icon-download" aria-hidden="true"></span>' . $linkText . '</button></joomla-toolbar-button>';

        $document = Factory::getApplication()->getDocument();
        $scriptDeclaration = 'jQuery(function($){$("#toolbar").append(\'' . $proButton . '\');if(!$("#toolbar joomla-toolbar-button").hasClass("ms-auto")) {$("#toolbar-pro-validation").addClass("ms-auto");};})';
        $document->addScriptDeclaration($scriptDeclaration);

        $loaded = true;
    }

    /**
     * @return string
     * @since 5.0.0.0-PRO
     */
    private function getProSecurityToken(): string
    {
        static $proSecurityToken = null;

        if ($proSecurityToken !== null) {
            return $proSecurityToken;
        }

        require_once $this->tokenPath;

        return $proSecurityToken ?? '';
    }

    /**
     * @param string $proSecurityToken
     *
     * @return bool
     * @throws JsonException|Exception
     * @since 5.0.0.0-PRO
     */
    private function getProSecurityTokenStatus(string $proSecurityToken): bool
    {
        static $proSecurityTokenStatus = null;

        if ($proSecurityTokenStatus !== null) {
            return $proSecurityTokenStatus;
        }

        if (HttpFactory::getAvailableDriver(new Registry()) === false) {
            throw new RuntimeException(Text::_('PRO_SECURITY_TOKEN_EXCEPTION_NO_HTTP_DRIVER'));
        }

        $urlCheck = 'https://check.kubik-rubik.de/pro/' . $proSecurityToken;

        try {
            $proSecurityTokenRequest = HttpFactory::getHttp()->get($urlCheck);
        } catch (Exception $e) {
            throw new RuntimeException(Text::_('PRO_SECURITY_TOKEN_EXCEPTION_NO_CONNECTION'));
        }

        $returnCode = $proSecurityTokenRequest->code;

        if ($returnCode !== 200) {
            throw new RuntimeException(Text::_('PRO_SECURITY_TOKEN_EXCEPTION_RESPONSE_CODE'));
        }

        $body = $proSecurityTokenRequest->body;

        if ($body === '') {
            throw new RuntimeException(Text::_('PRO_SECURITY_TOKEN_EXCEPTION_NO_BODY'));
        }

        $body = json_decode($body, true, 512, JSON_THROW_ON_ERROR);
        $proSecurityTokenStatus = true;

        return $body['status'] === true;
    }
}
