<?php

/**
 * @copyright
 * @package    Easy Joomla Backup Pro - EJB for Joomla! 5.x
 * @author     Viktor Vogel <admin@kubik-rubik.de>
 * @version    5.0.1.0-PRO - 2024-01-29
 * @link       https://kubik-rubik.de/ejb-easy-joomla-backup
 *
 * @license    GNU/GPL
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

namespace KubikRubik\Component\EasyJoomlaBackup\Administrator\Helper;

defined('_JEXEC') || die('Restricted access');

use Exception;
use Joomla\CMS\{Factory, Component\ComponentHelper, Language\Text, Log\Log, Mail\MailerFactoryInterface, Table\Table, HTML\HTMLHelper};
use Joomla\Filesystem\{File, Folder};
use phpseclib3\Net\SFTP;
use RuntimeException;
use ZipArchive;

use function defined;
use function function_exists;
use function in_array;

/**
 * Class EasyJoomlaBackupHelper
 *
 * @package EasyJoomlaBackup
 * @version 5.0.1.0-PRO
 * @since   5.0.0.0-PRO
 */
class EasyJoomlaBackupHelper
{
    /**
     * @var string EASYJOOMLABACKUP_VERSION
     * @version 5.0.1.0-PRO
     * @since   5.0.0.0-PRO
     */
    public const EASYJOOMLABACKUP_VERSION = '5.0.1.0-PRO';

    /**
     * @var string EASYJOOMLABACKUP_SYSTEM_NAME
     * @since 5.0.0.0-PRO
     */
    public const EASYJOOMLABACKUP_SYSTEM_NAME = 'com_easyjoomlabackup';

    /**
     * @var string BACKUP_TYPE_DATABASE
     * @since 5.0.0.0-PRO
     */
    public const BACKUP_TYPE_DATABASE = 'databasebackup';

    /**
     * @var string BACKUP_TYPE_FILE
     * @since 5.0.0.0-PRO
     */
    public const BACKUP_TYPE_FILE = 'filebackup';

    /**
     * @var string BACKUP_TYPE_FULL
     * @since 5.0.0.0-PRO
     */
    public const BACKUP_TYPE_FULL = 'fullbackup';

    /**
     * @var string MESSAGE_TYPE_MESSAGE
     * @since 5.0.0.0-PRO
     */
    public const MESSAGE_TYPE_MESSAGE = 'message';

    /**
     * @var string MESSAGE_TYPE_NOTICE
     * @since 5.0.0.0-PRO
     */
    public const MESSAGE_TYPE_NOTICE = 'notice';

    /**
     * @var string MESSAGE_TYPE_WARNING
     * @since 5.0.0.0-PRO
     */
    public const MESSAGE_TYPE_WARNING = 'warning';

    /**
     * @var string MESSAGE_TYPE_ERROR
     * @since 5.0.0.0-PRO
     */
    public const MESSAGE_TYPE_ERROR = 'error';

    /**
     * @var string LIBZIP_VERSION_MIN
     * @since 5.0.0.0-PRO
     */
    public const LIBZIP_VERSION_MIN = '1.2.0';

    /**
     * @var int UPLOAD_LIMIT_DROPBOX_BYTES
     * @since 5.0.0.0-PRO
     */
    public const UPLOAD_LIMIT_DROPBOX_BYTES = 157286400;

    /**
     * @var int UPLOAD_SESSION_CHUNK_BYTES
     * @since 5.0.0.0-PRO
     */
    public const UPLOAD_SESSION_CHUNK_BYTES = 104857600;

    /**
     * @var string UPLOAD_DROPBOX_STATUS_NAME
     * @since 5.0.0.0-PRO
     */
    public const UPLOAD_DROPBOX_STATUS_NAME = 'dropboxUploadStatus';

    /**
     * @var string UPLOAD_SFTP_STATUS_NAME
     * @since 5.0.0.0-PRO
     */
    public const UPLOAD_SFTP_STATUS_NAME = 'sftpUploadStatus';

    /**
     * @var string EXTENSION_NAME
     * @since 5.0.0.0-PRO
     */
    public const EXTENSION_NAME = 'com_easyjoomlabackup';

    /**
     * @var string $sessionNamespace
     * @since 5.0.0.0-PRO
     */
    public static string $sessionNamespace = 'EasyJoomlaBackup';

    /**
     * @var array $messageQueue
     * @since 5.0.0.0-PRO
     */
    protected static array $messageQueue = [];

    /**
     * @var string $sessionName
     * @since 5.0.0.0-PRO
     */
    protected static string $sessionName = 'messageQueue';

    /**
     * @var string $sessionLibzipStatusName
     * @since 5.0.0.0-PRO
     */
    protected static string $sessionLibzipStatusName = 'libzipStatus';

    /**
     * Adds a message to the message queue
     *
     * @param string $message
     * @param string $type
     *
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    public static function addMessage(string $message, string $type = self::MESSAGE_TYPE_MESSAGE): void
    {
        $messageQueue = Factory::getApplication()->getSession()->get(self::$sessionName, [], self::$sessionNamespace);
        $messageQueue[] = compact('message', 'type');
        Factory::getApplication()->getSession()->set(self::$sessionName, $messageQueue, self::$sessionNamespace);
    }

    /**
     * Adds message from the message queue to the system message queue
     *
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    public static function showMessages(): void
    {
        $messageQueue = self::getMessages();

        if (!empty($messageQueue)) {
            foreach ($messageQueue as $message) {
                Factory::getApplication()->enqueueMessage($message['message'], $message['type']);
            }
        }
    }

    /**
     * Gets messages from the message queue
     *
     * @return array
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    public static function getMessages(): array
    {
        $messageQueue = Factory::getApplication()->getSession()->get(self::$sessionName, [], self::$sessionNamespace);
        Factory::getApplication()->getSession()->clear(self::$sessionName, self::$sessionNamespace);

        return $messageQueue;
    }

    /**
     * Check whether required function to encrypt the files is available
     * Since PHP >= 7.2.0, PECL zip >= 1.14.0 (Libzip >= 1.2.0)
     *
     * @return bool
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    public static function checkEncryptArchive(): bool
    {
        $zipObject = new ZipArchive();

        if (!method_exists($zipObject, 'setEncryptionName')) {
            return false;
        }

        $encryptBackupArchive = (bool)ComponentHelper::getParams(self::EXTENSION_NAME)->get('encryptBackupArchive', false);

        if (!$encryptBackupArchive) {
            return false;
        }

        if (!self::checkLibZipCompatibility()) {
            return false;
        }

        return true;
    }

    /**
     * Checks the Libzip compatibility for the encryption process, result is
     * stored in the session to avoid the resource intensive call on each request
     *
     * @return bool
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    private static function checkLibZipCompatibility(): bool
    {
        $libzipStatusSession = Factory::getApplication()->getSession()->get(self::$sessionLibzipStatusName, null, self::$sessionNamespace);

        if ($libzipStatusSession !== null) {
            return (bool)$libzipStatusSession;
        }

        ob_start();
        date_default_timezone_set('UTC');
        phpinfo(INFO_MODULES);
        $phpInfo = ob_get_clean();

        $patternLibzip = '@libzip.*version[^\d]*(\d\.\d\.?\d?)@i';

        if (preg_match($patternLibzip, $phpInfo, $match)) {
            if (!empty($match[1]) && version_compare($match[1], self::LIBZIP_VERSION_MIN, 'ge')) {
                Factory::getApplication()->getSession()->set(self::$sessionLibzipStatusName, true, self::$sessionNamespace);

                return true;
            }
        }

        Factory::getApplication()->getSession()->set(self::$sessionLibzipStatusName, false, self::$sessionNamespace);

        return false;
    }

    /**
     * Returns an array with all item IDs that are already stored in an associated 3rd party hosting service
     *
     * @return array
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    public static function getUploadStatus(): array
    {
        $uploadStatus = [
            'dropbox' => [],
            'sftp' => [],
        ];

        if (!in_array(true, self::isUploadAllowed(), true)) {
            return $uploadStatus;
        }

        $dropboxUploadStatus = self::getDropboxUploadStatus();

        if (!empty($dropboxUploadStatus)) {
            $uploadStatus['dropbox'] = $dropboxUploadStatus;
        }

        $sftpUploadStatus = self::getSftpUploadStatus();

        if (!empty($sftpUploadStatus)) {
            $uploadStatus['sftp'] = $sftpUploadStatus;
        }

        if (!empty($uploadStatus['dropbox']) || !empty($uploadStatus['sftp'])) {
            return $uploadStatus;
        }

        $uploadStatus['dropbox'] = self::resetDropboxUploadStatus();
        $uploadStatus['sftp'] = self::resetSftpUploadStatus();

        return $uploadStatus;
    }

    /**
     * Checks for all 3rd party hosting services whether the uploads can be executed
     *
     * @param string $type
     *
     * @return array|bool
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    public static function isUploadAllowed(string $type = ''): array|bool
    {
        $uploadAllowed = [
            'dropbox' => false,
            'sftp' => false,
        ];

        if (Factory::getApplication()->getIdentity()->authorise('easyjoomlabackup.dropboxUpload', self::EXTENSION_NAME)) {
            $dropboxUploadAccessToken = ComponentHelper::getParams(self::EXTENSION_NAME)->get('dropboxUploadAccessToken');

            if (!empty($dropboxUploadAccessToken)) {
                $uploadAllowed['dropbox'] = true;
            }
        }

        if (Factory::getApplication()->getIdentity()->authorise('easyjoomlabackup.sftUpload', self::EXTENSION_NAME)) {
            $sftpUploadHost = ComponentHelper::getParams(self::EXTENSION_NAME)->get('sftpUploadHost', '');
            $sftpUsername = ComponentHelper::getParams(self::EXTENSION_NAME)->get('sftpUploadUsername', '');
            $sftpPassword = ComponentHelper::getParams(self::EXTENSION_NAME)->get('sftpUploadPassword', '');

            if ($sftpUploadHost !== '' && $sftpUsername !== '' && $sftpPassword !== '') {
                $uploadAllowed['sftp'] = true;
            }
        }

        if ($type !== '' && isset($uploadAllowed[$type])) {
            return $uploadAllowed[$type];
        }

        return $uploadAllowed;
    }

    /**
     * Gets the Dropbox upload status
     *
     * @return array
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    public static function getDropboxUploadStatus(): array
    {
        $dropboxUploadStatus = ComponentHelper::getParams(self::EXTENSION_NAME)->get(self::UPLOAD_DROPBOX_STATUS_NAME);

        if (empty($dropboxUploadStatus)) {
            return [];
        }

        return json_decode($dropboxUploadStatus, true, 512, JSON_THROW_ON_ERROR);
    }

    /**
     * Gets the SFTP upload status
     *
     * @return array
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    public static function getSftpUploadStatus(): array
    {
        $sftpUploadStatus = ComponentHelper::getParams(self::EXTENSION_NAME)->get(self::UPLOAD_SFTP_STATUS_NAME);

        if (empty($sftpUploadStatus)) {
            return [];
        }

        return json_decode($sftpUploadStatus, true, 512, JSON_THROW_ON_ERROR);
    }

    /**
     * Resets the Dropbox upload status - loads fresh information from the Dropbox API
     *
     * @return array
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    public static function resetDropboxUploadStatus(): array
    {
        $dopboxUploadStatus = [];

        if (!self::isUploadAllowed('dropbox')) {
            return $dopboxUploadStatus;
        }

        $dropboxUploadAccessToken = ComponentHelper::getParams(self::EXTENSION_NAME)->get('dropboxUploadAccessToken');

        if (empty($dropboxUploadAccessToken)) {
            return $dopboxUploadStatus;
        }

        $dropbox = new DropboxHelper($dropboxUploadAccessToken);
        $listFolder = $dropbox->listFolder();
        $items = self::getItemsIdName();

        if (!empty($listFolder)) {
            foreach ($listFolder as $listEntry) {
                foreach ($items as $item) {
                    if ($listEntry['name'] === $item['name']) {
                        $dopboxUploadStatus[] = $item['id'];

                        break;
                    }
                }
            }
        }

        self::setDropboxUploadStatus($dopboxUploadStatus);

        return $dopboxUploadStatus;
    }

    /**
     * Gets the IDs and names of all database entries
     *
     * @return array
     * @since 5.0.0.0-PRO
     */
    private static function getItemsIdName(): array
    {
        $db = Factory::getContainer()->get('DatabaseDriver');
        $query = $db->getQuery(true);
        $query->select('id')->select('name');
        $query->from($db->quoteName('#__easyjoomlabackup'));
        $db->setQuery($query);

        return $db->loadAssocList();
    }

    /**
     * Sets the Dropbox upload status
     *
     * @param array $dropboxListStatus
     *
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    public static function setDropboxUploadStatus(array $dropboxListStatus): void
    {
        self::updateUploadStatus(self::UPLOAD_DROPBOX_STATUS_NAME, $dropboxListStatus);
    }

    /**
     * Updates the upload status in the settings of the component
     *
     * @param string $uploadType
     * @param array  $listStatus
     *
     * @return void
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    private static function updateUploadStatus(string $uploadType, array $listStatus): void
    {
        $component = ComponentHelper::getComponent(self::EXTENSION_NAME);
        $params = $component->getParams();

        if (!empty($params) && !empty($component->id)) {
            $params->set($uploadType, json_encode($listStatus, JSON_THROW_ON_ERROR));

            $table = Table::getInstance('extension');
            $table->load($component->id);
            $table->bind(['params' => $params->toString()]);

            if (!$table->check()) {
                return;
            }

            $table->store();
        }
    }

    /**
     * Resets the SFTP upload status - loads fresh information from the connected server
     *
     * @return array
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    public static function resetSftpUploadStatus(): array
    {
        $sftpUploadStatus = [];

        if (!self::isUploadAllowed('sftp')) {
            return $sftpUploadStatus;
        }

        $sftp = self::getSftp();

        if ($sftp === null) {
            return $sftpUploadStatus;
        }

        $sftpUploadPath = ComponentHelper::getParams(self::EXTENSION_NAME)->get('sftpUploadPath', '');
        $sftp->chdir($sftpUploadPath);

        $listFolder = $sftp->rawlist();
        $items = self::getItemsIdName();

        if (!empty($listFolder)) {
            foreach ($listFolder as $listEntry) {
                foreach ($items as $item) {
                    if ($listEntry['filename'] === $item['name']) {
                        $sftpUploadStatus[] = $item['id'];

                        break;
                    }
                }
            }
        }

        self::setSftpUploadStatus($sftpUploadStatus);

        return $sftpUploadStatus;
    }

    /**
     * Gets the SFTP upload object if the connection is established successfully
     *
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    public static function getSftp(): ?SFTP
    {
        if (!self::isUploadAllowed('sftp')) {
            return null;
        }

        $sftpUploadHost = ComponentHelper::getParams(self::EXTENSION_NAME)->get('sftpUploadHost', '');
        $sftpUsername = ComponentHelper::getParams(self::EXTENSION_NAME)->get('sftpUploadUsername', '');
        $sftpPassword = ComponentHelper::getParams(self::EXTENSION_NAME)->get('sftpUploadPassword', '');

        require_once __DIR__ . '/../Libraries/Autoload.php';

        $sftp = new SFTP($sftpUploadHost);
        $sftp->login($sftpUsername, $sftpPassword);

        $sftpUploadPath = ComponentHelper::getParams(self::EXTENSION_NAME)->get('sftpUploadPath', '');
        $sftp->mkdir($sftpUploadPath, -1, true);
        $sftp->chdir($sftpUploadPath);

        return $sftp;
    }

    /**
     * Sets the SFTP upload status
     *
     * @param array $sftpListStatus
     *
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    public static function setSftpUploadStatus(array $sftpListStatus): void
    {
        self::updateUploadStatus(self::UPLOAD_SFTP_STATUS_NAME, $sftpListStatus);
    }

    /**
     * Is the set backup storage location writable by PHP
     *
     * @return bool
     * @version 5.0.0.1-PRO
     * @since   5.0.0.0-PRO
     */
    public static function isBackupStorageLocationWritable(): bool
    {
        $backupStorageLocationPath = self::getBackupStorageLocation();

        if (!is_dir($backupStorageLocationPath)) {
            try {
                Folder::create($backupStorageLocationPath);
            } catch (Exception $e) {
                return false;
            }
        }

        if (is_dir($backupStorageLocationPath) && !is_file($backupStorageLocationPath . '/.htaccess')) {
            File::write($backupStorageLocationPath . '/.htaccess', 'Deny from all');
        }

        return is_writable(self::getBackupStorageLocation());
    }

    /**
     * Gets the backup storage location from the configuration
     *
     * @param bool $trailingSlash
     * @param bool $root
     *
     * @return string
     * @version 5.0.1.0-PRO
     * @since   5.0.0.0-PRO
     */
    public static function getBackupStorageLocation(bool $trailingSlash = false, bool $root = false): string
    {
        $backupStorageLocation = ComponentHelper::getParams(self::EXTENSION_NAME)->get('backupStorageLocation', 'administrator/components/com_easyjoomlabackup/backups');

        if (str_contains($backupStorageLocation, ':\\')) {
            if (!$trailingSlash) {
                $backupStorageLocation = rtrim($backupStorageLocation, '/');
            }

            if ($trailingSlash && !str_ends_with($backupStorageLocation, '/')) {
                $backupStorageLocation .= '/';
            }

            return $backupStorageLocation;
        }

        if (strncmp($backupStorageLocation, '/', 1) !== 0) {
            $backupStorageLocation = '/' . $backupStorageLocation;
        }

        if (!$trailingSlash) {
            $backupStorageLocation = rtrim($backupStorageLocation, '/');
        }

        if ($trailingSlash && !str_ends_with($backupStorageLocation, '/')) {
            $backupStorageLocation .= '/';
        }

        if ($root) {
            return $backupStorageLocation;
        }

        return JPATH_ROOT . $backupStorageLocation;
    }

    /**
     * Send the notification mail with the provided data
     *
     * @param array  $mailAddresses
     * @param bool   $status
     * @param string $urlRoot
     * @param array  $data
     * @param string $outputType
     *
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    public static function sendNotificationMail(array $mailAddresses, bool $status, string $urlRoot, array $data, string $outputType = 'default'): void
    {
        $mail = Factory::getContainer()->get(MailerFactoryInterface::class)->createMailer();
        $mail->isHtml();
        $app = Factory::getApplication();
        $siteName = $app->get('sitename');
        $app->getLanguage()->load('com_easyjoomlabackup', JPATH_ADMINISTRATOR);

        $mail->setSubject(self::getMailSubject($status, $siteName));
        $mail->setBody(self::getMailBody($status, $urlRoot, $data, $siteName, $outputType));

        $mail->addRecipient($mailAddresses);
        $mail->setSender([$app->get('mailfrom'), $app->get('fromname')]);

        if ($app->get('mailer') === 'sendmail') {
            $mail->useSendmail($app->get('sendmail'));
        } elseif ($app->get('mailer') === 'smtp') {
            $mail->useSmtp($app->get('smtpauth'), $app->get('smtphost'), $app->get('smtpuser'), $app->get('smtppass'), $app->get('smtpsecure'), $app->get('smtpport'));
        }

        $mail->Send();
    }

    /**
     * Gets the mail subject
     *
     * @param bool   $status
     * @param string $siteName
     *
     * @return string
     * @since 5.0.0.0-PRO
     */
    private static function getMailSubject(bool $status, string $siteName): string
    {
        if (!$status) {
            return Text::sprintf('COM_EASYJOOMLABACKUP_NOTIFICATION_MAIL_SUBJECT_ERROR', $siteName);
        }

        return Text::sprintf('COM_EASYJOOMLABACKUP_NOTIFICATION_MAIL_SUBJECT_SUCCESS', $siteName);
    }

    /**
     * Gets the mail body
     *
     * @param bool   $status
     * @param string $urlRoot
     * @param array  $data
     * @param string $siteName
     * @param string $outputType
     *
     * @return string
     * @throws Exception
     * @since   5.0.0.0-PRO
     */
    private static function getMailBody(bool $status, string $urlRoot, array $data, string $siteName, string $outputType = 'default'): string
    {
        if ($outputType === 'json') {
            $data['status'] = true;

            if (!$status) {
                $data['status'] = false;
            }

            return json_encode($data, JSON_THROW_ON_ERROR);
        }

        $emailBody = '';

        if ($status) {
            $emailBody .= '<p>' . Text::sprintf('COM_EASYJOOMLABACKUP_NOTIFICATION_MAIL_BODY_SUCCESS', $siteName, $urlRoot) . '</p>';
        } else {
            $emailBody .= '<p>' . Text::sprintf('COM_EASYJOOMLABACKUP_NOTIFICATION_MAIL_BODY_ERROR', $siteName, $urlRoot) . '</p>';
        }

        if (!empty($data)) {
            $emailBody .= '<p><strong>' . Text::_('COM_EASYJOOMLABACKUP_NOTIFICATION_MAIL_BODY_DATA') . '</strong></p>';

            foreach ($data as $datumKey => $datumValue) {
                if ($datumKey === 'dateLocal') {
                    continue;
                }

                if ($datumKey === 'date') {
                    $datumValue = HTMLHelper::_('date', $datumValue, Text::_('DATE_FORMAT_LC2'), false);
                } elseif ($datumKey === 'type') {
                    if ($datumValue === 'fullbackup') {
                        $datumValue = Text::_('COM_EASYJOOMLABACKUP_FULLBACKUP');
                    } elseif ($datumValue === 'databasebackup') {
                        $datumValue = Text::_('COM_EASYJOOMLABACKUP_DATABASEBACKUP');
                    } elseif ($datumValue === 'filebackup') {
                        $datumValue = Text::_('COM_EASYJOOMLABACKUP_FILEBACKUP');
                    }
                } elseif ($datumKey === 'name') {
                    $datumValue = preg_replace('@_[^_]*\.zip@', '_[...].zip', $datumValue);
                } elseif ($datumKey === 'size') {
                    if ($datumValue > 1048576) {
                        $datumValue = number_format($datumValue / 1048576, 2, ',', '.') . ' MB';
                    } else {
                        $datumValue = number_format($datumValue / 1024, 2, ',', '.') . ' KB';
                    }
                } elseif ($datumKey === 'duration') {
                    if ($datumValue > 60) {
                        $datumValue = floor($datumValue / 60) . ' m ' . ($datumValue % 60) . ' s';
                    } elseif ($datumValue > 0) {
                        $datumValue .= ' s';
                    }
                }

                $emailBody .= '<p><strong>' . Text::_('COM_EASYJOOMLABACKUP_NOTIFICATION_MAIL_BODY_DATA_' . strtoupper($datumKey)) . '</strong><br />' . $datumValue . '</p>';
            }
        }

        if ($status) {
            $emailBody .= '<p>' . Text::_('COM_EASYJOOMLABACKUP_NOTIFICATION_MAIL_FOOTER_SUCCESS') . '</p>';
        } else {
            $emailBody .= '<p>' . Text::_('COM_EASYJOOMLABACKUP_NOTIFICATION_MAIL_FOOTER_ERROR') . '</p>';
        }

        $emailBody .= self::getFooterNotification();

        return $emailBody;
    }

    /**
     * Gets the footer with the version number for the notification email
     *
     * @return string
     * @since 5.0.0.0-PRO
     */
    public static function getFooterNotification(): string
    {
        return '<p style="text-align: right"><small>' . Text::sprintf('COM_EASYJOOMLABACKUP_NOTIFICATION_MAIL_FOOTER_POWERED', self::EASYJOOMLABACKUP_VERSION) . '</small></p>';
    }

    /**
     * Gets the footer link with the version number
     *
     * @return string
     * @since 5.0.0.0-PRO
     */
    public static function getFooter(): string
    {
        return '<div style="text-align: center; margin-top: 10px;"><p>' . Text::sprintf('COM_EASYJOOMLABACKUP_VERSION', self::EASYJOOMLABACKUP_VERSION) . '</p></div>';
    }

    /**
     * Removes a trailing comma from a string
     *
     * @param string $string
     *
     * @return string
     * @since 5.0.0.0-PRO
     */
    public static function removeTrailingComma(string $string): string
    {
        if (str_ends_with($string, ',')) {
            $string = substr($string, 0, -1);
        }

        return $string;
    }

    /**
     * Gets the database dump header information
     *
     * @return string
     * @since 5.0.0.0-PRO
     */
    public static function getDatabaseDumpHeader(): string
    {
        $data = '-- Easy Joomla Backup Pro for Joomla! 5.x - SQL Dump' . "\n";
        $data .= '-- Author: Viktor Vogel' . "\n";
        $data .= '-- Project: Kubik-Rubik Joomla! Extensions' . "\n";
        $data .= '-- Project page: https://kubik-rubik.de/ejb-easy-joomla-backup' . "\n";
        $data .= '-- License: GNU/GPL - https://www.gnu.org/licenses/gpl.html' . "\n\n";

        return $data;
    }

    /**
     * Opens a ZIP file archive
     *
     * @param string $backupPath
     *
     * @return ZipArchive
     * @since 5.0.0.0-PRO
     */
    public static function openZipArchive(string $backupPath): ZipArchive
    {
        $zipArchive = new ZipArchive();
        $zipFlag = ZipArchive::CREATE;

        if (is_file($backupPath) && filesize($backupPath) === 0) {
            $zipFlag = ZipArchive::OVERWRITE;
        }

        if ($zipArchive->open($backupPath, $zipFlag) !== true) {
            throw new RuntimeException('Error: Zip file could not be created!');
        }

        return $zipArchive;
    }

    /**
     * Closes a ZIP file archive
     *
     * @param ZipArchive $zipArchive
     *
     * @return void
     * @since 5.0.0.0-PRO
     */
    public static function closeZipArchive(ZipArchive $zipArchive): void
    {
        $zipArchive->close();
    }

    /**
     * Logs a message using Joomla!'s core log system
     *
     * @param $message
     *
     * @return void
     * @since 5.0.0.0-PRO
     */
    public static function log($message): void
    {
        Log::add($message, Log::ERROR, self::EASYJOOMLABACKUP_SYSTEM_NAME);
    }

    /**
     * Increases the server limits
     *
     * @return void
     * @since 5.0.0.0-PRO
     */
    public static function increaseServerLimits(): void
    {
        if (function_exists('set_time_limit')) {
            set_time_limit(0);
        }

        if (function_exists('ini_set')) {
            ini_set('memory_limit', -1);
            ini_set('error_reporting', 0);
        }
    }
}
